/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "sync_goertzel.h"
#include "atom.h"
#include "qbio.h"
#include "util.h"

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

qb_err_t qb_sync_freq (s32_t sample_rate,
                       qb_span_t *spans,
                       s32_t num_spans,
                       float *goertz[2],
                       float hf_goertz_test_multiplier,
                       float max_powerX,
                       qb_inspect_t *inspect,
                       u8_t force_even_cycled_run,
                       u8_t verbose) {

  qb_err_t e;
  s32_t sn;
  
  e = QB_E_OK;
  
  for (sn=0; sn < num_spans; sn++) {
  
    qb_span_t *span;
    s64_t i;
    double nominal_cyclen_smps_f;
    s64_t last_change_smps;
    s32_t span_cycs_alloc;
    u8_t current_value;
    s64_t run_len_smps;
    qb_span_t *prev_span; // for qb_is_squawk()
    s16_t *inspect_power_flips_buf;
    s64_t half_2400_cycle_smps_nom;
    
    span      = spans + sn;
    prev_span = (sn>0) ? (span-1) : NULL;
    
    half_2400_cycle_smps_nom = (s64_t) roundf (((float) sample_rate) / (2.0f * QB_FREQ_2 * span->speed));

//printf("half_2400_cycle_smps_nom = %lld\n", half_2400_cycle_smps_nom);

    inspect_power_flips_buf = NULL;
    
    if (inspect->enabled) {
      inspect_power_flips_buf = qb_malloc (sizeof(s16_t) * span->len);
      if (NULL == inspect_power_flips_buf) {
        fprintf(QB_ERR, "E: Out of memory allocating frequency sync power flips buffer.\n");
        e = QB_E_MALLOC;
        break;
      }
      memset (inspect_power_flips_buf, 0, sizeof(s16_t) * span->len);
    }
    
    if (QB_SPAN_TYPE_DATA != span->type) {
      // just skip this span, but if inspect is enabled,
      // we need to write out a padding section to the inspect file
      if (inspect->enabled) {
        e = qb_inspect_append_s16_buf (inspect->files + QB_INSPECT_FILE_IX_FREQ_POWER_FLIPS,
                                       inspect_power_flips_buf, // all zeros
                                       span->len);
        qb_free(inspect_power_flips_buf);
        inspect_power_flips_buf = NULL;
        if (QB_E_OK != e) { break; }
      }
      continue; // data spans only
    }
    
    nominal_cyclen_smps_f = ((double) sample_rate) / (QB_FREQ_2 * span->speed);
    last_change_smps = span->start;
    span_cycs_alloc = 0;
    current_value = 1;
    
    for (i = span->start; i < (span->start + span->len); i++) {
  
      float p0, p1;
      u8_t flipped;
      double run_len_cycles_f;
      double actual_cycle_len_smps_f; // REQUIRES DOUBLE PRECISION
      s32_t run_len_cycles;
      s32_t run_len_cycles_even;
      double current_pos_smps_f; // REQUIRES DOUBLE PRECISION
      s32_t k;
      s64_t q;
      
      //if (QB_SPAN_TYPE_DATA != span->type) {
      //  continue; // data spans only
      //}

      p0 = goertz[0][i] / max_powerX;
      p1 = (goertz[1][i] * hf_goertz_test_multiplier) / max_powerX;
      flipped = 0;
      
      if (1 == current_value) {
        if (p0 > p1) { flipped = 1; }
      } else {
        if (p1 > p0) { flipped = 1; }
      }
        
      /*
        -------+                                          +------- goertzel trace
               |                                          |
           1   |                    0                     |   1
               |                                          |
               +------------------------------------------+

               |<---           run_len_smps           --->|

               <nom. cy>|<nom. cy>|<nom. cy>|<nom. cy>|<rem    4.4 nominal cycs, round to 4, so
               
               <        >|<        >|<        >|<        >|    4 fitted cycles, each len actual_cycle_len_smps_f
               
                    ^          ^          ^          ^         create sampling instants here, at half way points
                    |          |          |          |
               A    C1         C2         C3         C4        position A is last_change_smps
                                                               C1 is last_change_smps + (0.5 x actual_cycle_len_smps_f)
                                                               etc.
      */
      
      run_len_smps = i - last_change_smps;
      
      // protect against high frequency oscillations on power0 and power1
      // which can cause a false double-crossing within a few samples
      
//printf("run_len_smps = %lld, half_2400_cycle_smps_nom = %lld\n", run_len_smps, half_2400_cycle_smps_nom);
      
      if (flipped && (run_len_smps < half_2400_cycle_smps_nom)) {
//printf("false flip\n");
        flipped = 0; // false flip, cancel it
        // move last change instant into the centre of this "fuzzy" period
        last_change_smps += ((run_len_smps) / 2);
        // need to fiddle already-written inspect values though
        if (inspect->enabled) {
          //inspect_power_flips_buf[i - span->start] = (p1 > p0) ? 20000 : 2000;
          for (q=0; q < ((run_len_smps) / 2); q++) {
            inspect_power_flips_buf[(q-1) + last_change_smps - span->start] = ( ! current_value ) ? 20000 : 2000;
          }
        }
      } else {
//printf("not a false flip\n");
      }
      
      
      if (flipped) {
        // flip value
        current_value = ((p1 > p0) ? 1 : 0);
      }
      
      if (inspect->enabled) {
        inspect_power_flips_buf[i - span->start] = current_value ? 20000 : 2000;
      }
      
      if (flipped) {
      
//printf("run_len_smps = %lld\n", run_len_smps);
        
        run_len_cycles_f = ((double) run_len_smps) / nominal_cyclen_smps_f;
        
        // hmm ... well, we can force run_len_cycles to be an even number,
        // which will fix loss of sync over very long (e.g. 18-cycle) runs
        // this is the most serious problem with the frequency sync method:
        // e.g. electron_sphinx_adventure_sideA contains some 0 bytes,
        // which give you nine zeros in a row on the tape (start bit + 8 data bits);
        // and even with tape speed detection, we can't track these quite accurately
        // enough not to miscount the cycles.
        
        run_len_cycles = (s32_t) round (run_len_cycles_f);

        // minimum run length limit removed.
        // ERROR CORRECTION:
        if ( force_even_cycled_run && ! qb_is_squawk (span, prev_span) ) { //&& (run_len_cycles > 2) ) {
          // force an even number of cycles
          run_len_cycles_even = 2 * (s32_t) round(run_len_cycles_f / 2.0);
          if (verbose && (run_len_cycles != run_len_cycles_even)) {
            fprintf(QB_ERR, "    W: [%lld] -> [%lld]: (%s %s) forced-even run [%d -> %d]\n",
                    last_change_smps,
                    i,
                    QB_CLI_S_SYNC_METHOD,
                    QB_CLI_S_SYNC_METHOD_GOERTZEL,
                    run_len_cycles,
                    run_len_cycles_even);
          }
          run_len_cycles = run_len_cycles_even; // replace orig. odd value
        }

        // use the *integer* number of cycles as the denominator
        // to get the actual length in smps of each cycle that we're going to use
        // (different from the theoretical one)
        
        // current_pos_smps_f is an accumulator; current_pos_smps_f and
        // actual_cycle_len_smps_f need double precision; we run into
        // problems otherwise.
        
        actual_cycle_len_smps_f = (((double) run_len_smps) / (double) run_len_cycles);
        current_pos_smps_f = ((double) last_change_smps) + (0.5 * actual_cycle_len_smps_f);
        
        // loop across the cycles that can be fitted into the period
        for (k=0; k < run_len_cycles; k++) {
        
          qb_atom_t c;
          
          qb_init_atom(&c);

          c.sample_num                  = (s64_t) round (current_pos_smps_f);

#ifdef QB_SANITY
          if ((sn>0) && (c.sample_num < (spans[sn-1].start + spans[sn-1].len))) {
            fprintf(QB_ERR, "B: %s: attempt to write data cycle prior to end of previous span\n"
                            "B: (prev span #%d, ends @ %lld, cyc @ %lld, last_change_smps %lld)\n",
                            QB_FUNC_M,
                            sn-1,
                            (spans[sn-1].start + spans[sn-1].len),
                            c.sample_num,
                            last_change_smps);
            e = QB_E_BUG;
            break;
          }
#endif
          
          //if (QB_SPAN_TYPE_DATA == span->type) {
          e = qb_append_atom (&c, &(span->atoms), &span_cycs_alloc, &(span->num_atoms));
          if (QB_E_OK != e) { break; }
          //}

          // ran into trouble with adjacent cycles having identical
          // sample numbers when a single-precision accumulator was used.
          // going back to double-precision fixed this, but this check
          // remains ...
#ifdef QB_SANITY
          if (span->num_atoms > 1) {
            if (span->atoms[span->num_atoms-1].sample_num == span->atoms[span->num_atoms-2].sample_num) {
              fprintf(QB_ERR, "B: %s: [%lld]: %s span %d: adjacent cycs w/same smpnum\n",
                      QB_FUNC_M,
                      span->atoms[span->num_atoms-1].sample_num,
                      qb_span_get_type_text(span),
                      sn);
              e = QB_E_BUG;
              break;
            }
          }
#endif
          
          if (QB_E_OK != e) {
            fprintf(QB_ERR, "E: Error adding cycle at mid_cycle smps %lld; run start(smps)=%lld, cycle %d/%d within run\n",
                    c.sample_num, i, k+1, run_len_cycles);
            //fprintf(QB_ERR, "E: cyc->samples_since_last = %lld\n", //; last_mid_cycle(smps) = %lld\n",
            //        c.samples_since_last); //, last_mid_cycle_smps);
            e = QB_E_BUG;
            break;
          }
          
          current_pos_smps_f  += actual_cycle_len_smps_f;
          
        } // next fitted cycle within the run
        
        if (QB_E_OK != e) { break; }
       
        last_change_smps = i;
        
      } // end if (flipped)
      
      /*
#ifdef QB_SANITY
      if (i < span->start) {
        fprintf(QB_ERR, "B: span %d: we're off the beginning of the span somehow?! (%lld vs %lld)\n",
                span->ix, i, span->start);
        return QB_E_BUG;  // FIXME: free existing cycles?
      }
#endif
  */

    } // next sample
    
    if (QB_E_OK != e) {
      qb_free(inspect_power_flips_buf);
      inspect_power_flips_buf = NULL;
      break;
    }
    
    if (inspect->enabled) {
      e = qb_inspect_append_s16_buf (inspect->files + QB_INSPECT_FILE_IX_FREQ_POWER_FLIPS,
                                     inspect_power_flips_buf,
                                     span->len);
      qb_free(inspect_power_flips_buf);
      inspect_power_flips_buf = NULL;
      if (QB_E_OK != e) { break; }
    }
    
//exit(0);
    
    //if (display_progress) { qb_update_meter (sn, num_spans, 1.0f, 0); }
    
  } // next span
  
  //if (display_progress) { qb_hide_meter(0); }
  //printf("done.\n");
  
  // on error, clean up cycs on all spans
  for (sn=0; (QB_E_OK != e) && (sn < num_spans); sn++) {
    qb_span_t *span;
    span = spans + sn;
    if (NULL != span->atoms) {
      qb_free(span->atoms);
      span->num_atoms = 0;
      span->atoms = NULL;
    }
  }
  
  return e;

}




