/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#ifndef QB_QUADBIKE_H
#define QB_QUADBIKE_H

#include "qb_types.h"

#include "build.h"
#include "cli_stgs.h"
#include "inspect.h"

#ifdef QB_WINDOWS
#include "Windows.h"
#endif

#include "qb_err.h"

// maximum filesize; let's limit it to 2 hours of audio
#define QB_MAX_IPFILE_SAMPLES (2LL * 60 * 60 * 48000)
// now in bytes; assume stereo & double precision;
// this is ~5.5 GB of RAM (per file):
#define QB_MAX_IPFILE_LEN     (QB_MAX_IPFILE_SAMPLES * 8 * 2)

#define TEST_LOOP_FILTER_CUTOFF_HZ  900.0 //600.0
#define TEST_LOCK_FILTER_CUTOFF_HZ  50.0
#define TEST_LOOP_GAIN              0.2 //0.2

#define QB_FREQ_1 1201.9f                  // http://beebwiki.mdfs.net/Acorn_cassette_format#Signal
#define QB_FREQ_2 (2.0f * QB_FREQ_1)

//#define QB_SILENT_POWER_FRACTION                  0.00015f
// 2.0.1: all thresholds (silent, signal, summed) are selectable now
#define QB_SILENT_THRESH_MIN     0.0001f
#define QB_SILENT_THRESH_MAX     0.1f
#define QB_SILENT_THRESH_DEFAULT 0.00015f

// we need some hysteresis, such that silent spans remain silent
// unless a relatively high power threshold is met. this
// is needed to avoid squawk spans starting earlier than
// they should, prepending a rogue zero-bit to the
// squawk span, which emulators interpret as a start
// bit. if this squawk span happens to transcribe to a '*'
// character, then we create a phantom block which isn't
// actually there.
//#define QB_POWER_FRACTION_SILENT_SPAN_OFF  0.1
#define QB_SIGNAL_THRESH_MIN     0.001 // 2.0.3: was 0.01 in 2.0.2
#define QB_SIGNAL_THRESH_MAX     0.2   // 2.0.3: was 0.5  in 2.0.2
#define QB_SIGNAL_THRESH_DEFAULT 0.02  // 2.0.3: was 0.1  in 2.0.2

// then we have the threshold for checking the integrated
// power across an entire signal span, and potentially
// converting it to a silent one.
#define QB_HUSH_THRESH_MIN      0.0 //0.00005
#define QB_HUSH_THRESH_MAX      0.00045 //0.00100
#define QB_HUSH_THRESH_DEFAULT  0.00002 //0.00015
#define QB_HUSH_RANGE           (QB_HUSH_THRESH_MAX - QB_HUSH_THRESH_MIN)
#define QB_HUSH_DEFAULT_FRAC    ((QB_HUSH_THRESH_DEFAULT - QB_HUSH_THRESH_MIN) / QB_HUSH_RANGE)
#define QB_HUSH_DEFAULT_PERCENT (100.0 * QB_HUSH_DEFAULT_FRAC)


// threshold for the PLL's (analogue, low-pass filtered) lock value
// at which the PLL is said to be locked
#define QB_LOCK_THRESH                   0.11
#define QB_LOCK_THRESH_TRANSIENT         0.3

#ifdef QB_WINDOWS
#define QB_PATHSEP '\\'
#else
#define QB_PATHSEP '/'
#endif

#define QB_PHASE_SCHEME_NONE         0
#define QB_PHASE_SCHEME_FIXED_AUTO   1
#define QB_PHASE_SCHEME_FIXED_0      2
#define QB_PHASE_SCHEME_FIXED_90     3
#define QB_PHASE_SCHEME_FIXED_180    4
#define QB_PHASE_SCHEME_FIXED_270    5
#define QB_PHASE_SCHEME_BLOCK_AUTO   6

/*
// NB: these are actually the same as the non-windoze
// versions below but in a separate section in case
// I need to change them later for whatever reason
#ifdef QB_WINDOWS
#ifndef FMT_U64
#define FMT_U64    "llu"
#endif
#ifndef FMT_U64HEX
#define FMT_U64HEX "llx"
#endif
#ifndef FMT_S64
#define FMT_S64    "lld"
#endif
#ifndef FMT_S64HEX
#define FMT_S64HEX "llx"
#endif
#else // ndef QB_WINDOWS
#ifndef FMT_U64
#define FMT_U64    "llu"
#endif
#ifndef FMT_U64HEX
#define FMT_U64HEX "llx"
#endif
#ifndef FMT_S64
#define FMT_S64    "lld"
#endif
#ifndef FMT_S64HEX
#define FMT_S64HEX "llx"
#endif
#endif // ndef QB_WINDOWS

#define FMT_SZT   "zu"
#define FMT_SZT_X "zx"
*/

#define QB_STR(X) #X
#define QB_XSTR(X) QB_STR(X)

#define QB_VERSION_STRING(MAJ,MIN,SUB) QB_APP_NAME " " QB_XSTR(MAJ) "." QB_XSTR(MIN) "." QB_XSTR(SUB)


#define QB_SYNC_METHOD_NONE       0
#define QB_SYNC_METHOD_WALK       1
#define QB_SYNC_METHOD_GOERTZEL   2
#define QB_SYNC_METHOD_PLL        3


#define QB_VERBOSITY_QUIET        -1
#define QB_VERBOSITY_DEFAULT      0
#define QB_VERBOSITY_NOISY        1

#define QB_PHASE_DETECT_METHOD_PLL  1
#define QB_PHASE_DETECT_METHOD_WALK 2

#define QB_SPAN_SHIFT_SEARCH_HALF_RANGE 2

#define QB_SCALE_2400_POWER_MIN 0.4
#define QB_SCALE_2400_POWER_MAX 3.5

// an input file might be stereo, so the
// maximum number of sources is double the
// maximum number of input files

typedef struct qb_s {

  u8_t sync_method;
  
  char *ipfile_path;
  u8_t *ipfile_buf;
  s64_t ipfile_buf_len;
  
  char *csw_opfile;
  char *tibet_opfile;
  char *tibet_z_opfile;
  u8_t allow_overwrite;
  
  u8_t do_filtering;
  
  s32_t sample_rate;
  
  u8_t phase_scheme; // BLOCK_AUTO etc
  u8_t phase_detect_method; // PLL or walk
  u8_t using_default_op_file;
  
  u8_t quieter;
  
  qb_inspect_t inspect;
  
  u8_t err_correction_strategy;
  
  u8_t have_forced_shift;
  s8_t forced_shift;
  
  u8_t forced_scaler_for_2400_power; // only for CLI summary ...
  float scaler_for_2400_power;
  
  u8_t verbose;
  u8_t normalise;
  
  u8_t use_right_channel;
  
  u8_t no_csw_polarity_correction;
  u8_t no_generic_odd_run_fix;
  
  double silence_on_frac;
  double signal_on_frac;
  double summed_silence_on_frac;

  u8_t find_best_hf_pwr;
  
  char *cwd;

} qb_t;

#endif // QB_QUADBIKE_H
