/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "qb_types.h"
#include "inspect.h"
#include "qbio.h"
#include "util.h"

#include <string.h>
#include <stdlib.h>

// libsndfile callbacks for writing inspect files (to disc):
static sf_count_t sfcb_inspect_tell (void *priv);
static sf_count_t sfcb_inspect_write (const void *q, sf_count_t count, void *priv);
static sf_count_t sfcb_inspect_read (void *q, sf_count_t count, void *priv);
static sf_count_t sfcb_inspect_seek (sf_count_t offset, int whence, void *priv);
static sf_count_t sfcb_inspect_get_filelen (void *priv);

//static qb_err_t qb_inspect_build_and_stat_path (qb_inspect_file_t *f, char *dir, const char *fn);
static qb_err_t qb_inspect_file_init (qb_inspect_file_t *insp_f, FILE *f);




// maximum allowed gap will be 5 minutes * 48K
#define HACK_DBG_WRITE_BITSAMPLES 14000000

// The second string in each element encodes the configuration
// under which each of these files will be produced:

// - first char:  sync method: P for PLL, F for freq, W for walk,
//                             or C, which means anywhere the PLL
//                             carriers are generated; so either PLL
//                             sync or walk mode but with --phase-det pll
// - second char: phase for which file is produced:
//                  0 for 0   degrees
//                  9 for 90  degrees
//                  1 for 180 degrees
//                  2 for 270 degrees
// - third char:  F if pre-filter is applied.

// Any character may also be " ", the wildcard, in which case
// the file will be produced for any configuration in that category.

// This allows us to predict at the start of execution
// which files will be written, so we can check that each
// file can be created right off the bat. We don't want to
// make the user wait until the point is reached where the
// files are actually written.

// see also constants in inspect.h.

const char *qb_inspect_filenames[QB_NUM_INSPECT_FILES][2] = {
  { "pll_output.wav",                       "P**"  },
  { "pll_integral.wav",                     "P**"  },
  { "pll_lock.wav",                         "P**"  },
  { "pll_quad_ref.wav",                     "P**"  },
  { "pll_carrier_1_freq_doubled.wav",       "C**"  },
  { "pll_carrier_2_mixed_phase_0.wav",      "C0*"  },
  { "pll_carrier_2_mixed_phase_90.wav",     "C9*"  },
  { "pll_carrier_2_mixed_phase_180.wav",    "C1*"  },
  { "pll_carrier_2_mixed_phase_270.wav",    "C2*"  },
  { "pll_carrier_3_filtered_phase_0.wav",   "C0*"  },
  { "pll_carrier_3_filtered_phase_90.wav",  "C9*"  },
  { "pll_carrier_3_filtered_phase_180.wav", "C1*"  },
  { "pll_carrier_3_filtered_phase_270.wav", "C2*"  },
  { "pll_cycle_times_pre_inserts.wav",      "P**"  },
  { "filtered.wav",                         "**F"  },
  { "cycle_times_walk_uninterpolated.wav",  "W**"  },
  { "span_types.wav",                       "***"  },
  { "power0.wav",                           "***"  },
  { "power1.wav",                           "***"  },
  { "confidence.wav",                       "***"  },
  { "cycle_values.wav",                     "***"  },
  { "cycle_times.wav",                      "***"  },
  { "span_tapespeeds.wav",                  "***"  },
  { "walk_features.wav",                    "W**"  },
  { "freq_power_flips.wav",                 "F**"  },
  { "walk_features_timing.wav",             "W**"  }, // only with def QB_EXPERIMENTAL_INSPECT_WRITE_FEATURES_TIMING
  { "spare_for_debugging.wav",              "   "  }
};


qb_err_t qb_inspect_init (qb_inspect_t *i, s32_t rate) {

  qb_err_t e;
  u8_t j;
  
  e = QB_E_OK;
  
  for (j=0; j < QB_NUM_INSPECT_FILES; j++) {
    qb_inspect_file_t *f;
    f = i->files + j;
    f->filename = qb_inspect_filenames[j][0];
    f->rate = rate;
    f->dir = i->dir;
    f->allow_overwrite = i->allow_overwrite;
  }
  
  return e;
  
}


static qb_err_t qb_inspect_file_init (qb_inspect_file_t *insp_f, FILE *f) {

  qb_err_t e;
  SF_VIRTUAL_IO sfvirt;
  SF_INFO sfinfo;
  
  e = qb_test_rate_legal (insp_f->rate);
  if ( QB_E_OK != e ) {
    fprintf(QB_ERR, "B: %s: bad sample rate %d.\n", QB_FUNC_M, insp_f->rate);
    return e;
  }
  
  //insp_f->priv.allow_overwrite = insp_f->allow_overwrite;
  insp_f->priv.rate  = insp_f->rate;
  sfvirt.read        = sfcb_inspect_read;
  sfvirt.get_filelen = sfcb_inspect_get_filelen;
  sfvirt.seek        = sfcb_inspect_seek;
  sfvirt.tell        = sfcb_inspect_tell;
  sfvirt.write       = sfcb_inspect_write;
  sfinfo.frames      = 0;
  sfinfo.samplerate  = insp_f->rate;
  sfinfo.channels    = 1;
  sfinfo.format      = SF_FORMAT_WAV | SF_FORMAT_PCM_U8;
  sfinfo.sections    = 0;
  sfinfo.seekable    = 0;
  insp_f->priv.f     = f;

  insp_f->sf = sf_open_virtual (&sfvirt, SFM_WRITE, &sfinfo, &(insp_f->priv));
  if (NULL == insp_f->sf) {
    fprintf(QB_ERR, "E: Could not open inspect file: %s\n", insp_f->priv.path);
    fprintf(QB_ERR, "   Error from libsndfile is \"%s\".\n", sf_strerror(NULL));
    //qb_free(insp_f->priv.path);
    //insp_f->priv.path = NULL;
    return QB_E_SNDFILE_OPEN_INSPECT;
  }

  return e;

}



qb_err_t qb_inspect_build_and_stat_path  (char *dir,
                                          const char *fn,
                                          char **path_out) {

  qb_stat_t qbstat;
  size_t pathlen, fnlen, fullpath_len;
  char *fnbuf;
  qb_err_t e;
  
//printf("qb_inspect_build_and_stat_path: %s/%s\n", dir, fn);

  if (NULL == dir) {
    fprintf(QB_ERR, "B: %s: directory is NULL!\n", QB_FUNC_M);
    return QB_E_BUG;
  }
  
  if (NULL == fn) {
    fprintf(QB_ERR, "B: %s: filename is NULL!\n", QB_FUNC_M);
    return QB_E_BUG;
  }

  pathlen = strlen(dir);
  fnlen   = strlen(fn);
  
  if (pathlen > 1000000) {
    fprintf(QB_ERR, "B: %s: ridiculous pathlen (%zu)\n", QB_FUNC_M, pathlen);
    e = QB_E_BUG;
    return e;
  }
  
  // +1 for directory separator (if needed), +1 for null-terminator
  fullpath_len = 2 + pathlen + fnlen;

  e = qb_stat_file (dir, &qbstat);
  if (QB_E_OK != e) {
    fprintf(QB_ERR, "W: Path for inspect files was not found: %s\n", dir);
    return e;
  }
  
  if ( ! qbstat.is_dir ) {
    fprintf(QB_ERR, "W: Path for inspect files is not a directory: %s\n", dir);
    return QB_E_INSPECT_DIR_NOT_DIR;
  }
  
  fnbuf = qb_malloc(fullpath_len);
  if (NULL == fnbuf) {
    fprintf(QB_ERR, "E: Out of memory allocating debug file path.\n");
    return QB_E_MALLOC;
  }
  
  // copy containing directory name to fnbuf
  memcpy(fnbuf, dir, pathlen);
  
  // if no directory separator specified, add it
  if (fnbuf[pathlen-1] != QB_PATHSEP) {
    fnbuf[pathlen] = QB_PATHSEP;
    pathlen++;
  }
  
  // append filename
  memcpy(fnbuf + pathlen, fn, 1 + fnlen);
  //printf("  inspect -> %s\n", fnbuf);
  
  //memset(sfpriv, 0, sizeof(qb_libsndfile_inspect_privdata_t));
  //sfpriv->path = fnbuf;
  
//printf("qb_inspect_build_and_stat_path: path is %s\n", sfpriv->path);
  
//  f->priv.allow_overwrite = allow_overwrite;

  *path_out = fnbuf;
  
  return QB_E_OK;
  
}


qb_err_t qb_inspect_append_halfbit_values (qb_span_t *span0,
                                           qb_span_t *span1,
                                           s32_t span0_ix,
                                           qb_inspect_file_t *f,
                                           s16_t *previous_atom_value_inout,
                                           s64_t *prev_smpnum_inout) {

  s64_t j;
  s32_t i;
  s16_t *buf;
  qb_err_t e;
  s64_t bufpos;
  s64_t len;
  qb_atom_t *cycs;
  
  cycs = span0->atoms;
  
  // have a subsequent span?
  if ((NULL != span1) && (NULL != span1->atoms)) {
    // limit is the *next* span's first cycle
    len = span1->atoms[0].sample_num - *prev_smpnum_inout;
  } else {
    // final span
    // use the approx span length as limit instead
    len = span0->len;
//printf("len = %lld\n", len);
    //len = cycs[span0->num_atoms-1].sample_num - *prev_smpnum_inout;
  }
  /*else {
    // everything is bad
    fprintf(QB_ERR, "W: qb_inspect_append_halfbit_values: span #%d (%s) has no cycles?! this is bad.\n",
            span0_ix, qb_span_get_type_text(span0));
    len = span0->len;
  }*/
    
  
  buf = qb_malloc(sizeof(s16_t) * len);
  
  if (NULL == buf) {
    fprintf(QB_ERR, "E: Out of memory writing halfbit instants inspection file.\n");
    return QB_E_MALLOC;
  }
  
  memset(buf, 0, sizeof(s16_t) * len);
  
  bufpos = 0;
  e   = QB_E_OK;
  
  //printf("inspect bitsamples: span0 atom0: start %lld, ssl %lld\n",
  //       span0->atoms[0].sample_num, span0->atoms[0].samples_since_last);
  
  for (i=0; (i < span0->num_atoms) && (bufpos < len); i++) {
  
    s64_t samples_since_last;

    samples_since_last = cycs[i].sample_num - *prev_smpnum_inout;
    
    if (    (samples_since_last > HACK_DBG_WRITE_BITSAMPLES) ) {
         //|| (samples_since_last < 1) ) { // relax this ...
      fprintf(QB_ERR,
              "B: %s: [%lld]: span %d,%d, type %s,%s: cycs[%d].samples_since_last is bad (%lld)\n",
              QB_FUNC_M,
              cycs[i].sample_num,
              span0_ix,
              (NULL!=span1)?(span0_ix+1):0,
              qb_span_get_type_text(span0),
              (NULL!=span1)?qb_span_get_type_text(span1):"null",
              i,
              samples_since_last);
      qb_free(buf);
      return QB_E_BUG;
    }

    // write previous cycle ...
    for (j=0;
         (j < samples_since_last) && (bufpos < len);
         j++, bufpos++) {
      buf[bufpos] = *previous_atom_value_inout;
    }
    if (bufpos > 0) {
      buf[bufpos-1] = -5000; // delineate the end of the cycle
    }
    
    // store this cycle value to be written next time
    // (this needs to be persisted outside this function, too,
    // because of span/span partitions)
    if ('L' == cycs[i].value) {
      *previous_atom_value_inout = QB_DBG_HALFBIT_VALUE_LEADER;
    } else if ('S' == cycs[i].value) {
      *previous_atom_value_inout = QB_DBG_HALFBIT_VALUE_SILENCE;
    } else if ('0' == cycs[i].value) {
      *previous_atom_value_inout = QB_DBG_HALFBIT_VALUE_ZERO;
    } else {
      *previous_atom_value_inout = QB_DBG_HALFBIT_VALUE_ONE;
    }
    
    *prev_smpnum_inout = cycs[i].sample_num;
    
  } // next cycle
  
  if (bufpos > len) {
    fprintf(QB_ERR, "E: [%lld] qb_inspect_append_halfbit_values: somehow span %d, cycle %d "
                    "implies a smpnum >= orig. span len (%lld vs %lld)\n",
                    span0->atoms[i].sample_num, span0_ix, i, bufpos, len);
    e = QB_E_BUG;
  } else {
    if (NULL != span1) {
      e = qb_inspect_append_s16_buf (f, buf, bufpos);
    } else {
      // hack to pad final silent span out to the full extent of the input file
      // (was causing a unit test to fail)
      e = qb_inspect_append_s16_buf (f, buf, len);
    }
  }
  
  qb_free(buf);

  return e;
  
}



qb_err_t qb_inspect_append_span_types (qb_span_t *spans,
                                       s64_t full_input_len,
                                       s32_t num_spans,
                                       //s32_t rate,
                                       qb_inspect_t *inspect) {
                                       
  s32_t i;
  s64_t sn;
  qb_err_t e;
  
  sn=0;
  
  e = QB_E_OK;
  
  for (i=0; i < num_spans; i++) {
  
    qb_span_t *span;
    s64_t j;
    s16_t *buf;
    
    span = spans + i;
    
//if (i < 3) {
//  printf("span %d, start %lld, len %lld, type %s\n",
//         i, span->start, span->len, qb_span_get_type_text(span));
//}
    
    buf = qb_malloc (span->len * sizeof(s16_t));
    if (NULL == buf) {
      fprintf(QB_ERR, "Out of memory allocating spans debug file buffer.\n");
      return QB_E_MALLOC;
    }
    memset(buf, 0, sizeof(s16_t) * span->len); // make sure of it
    
    for (j=0; (j < span->len) && (sn < full_input_len); j++, sn++) {
      if (QB_SPAN_TYPE_LEADER == span->type) {
        buf[j] = -16384;
      } else if (QB_SPAN_TYPE_DATA == span->type) {
        buf[j] = 16384;
      }
      // silence is 0
    }
    
    e = qb_inspect_append_s16_buf (inspect->files + QB_INSPECT_FILE_IX_SPAN_TYPES,
                                   buf,
                                   span->len);
    
    qb_free(buf);
    buf = NULL;
    
    if (QB_E_OK != e) { break; }
    
  }
  
  return e;
  
}




#include <math.h>


qb_err_t qb_inspect_append_span_goertzel_power (float *goertzel[2],
                                                s64_t srclen_smps,
                                                float max_powerX,
                                                float max_confidence,
                                                qb_inspect_t *inspect) {
                                                
  s64_t sn;
  s16_t *p0_n, *p1_n, *confi_n;
  qb_err_t e1, e2, e3; //, e;
  qb_inspect_file_t *isf_p0, *isf_p1, *isf_c;
  
  p0_n    = NULL;
  p1_n    = NULL;
  confi_n = NULL;

  // blah; we have to do this because we need to
  // normalise and scale the power, and we want to write the
  // whole lot out in big chunks instead of just
  // using fputc() which is horribly slow:
  p0_n    = qb_malloc (sizeof(s16_t) * srclen_smps);
  p1_n    = qb_malloc (sizeof(s16_t) * srclen_smps);
  confi_n = qb_malloc (sizeof(s16_t) * srclen_smps);

  if (    (NULL == p0_n)
       || (NULL == p1_n)
       || (NULL == confi_n) ) {
    fprintf(QB_ERR, "E: Out of memory allocating power debug file buffers.\n");
    if (NULL != p0_n)    { qb_free(p0_n);    }
    if (NULL != p1_n)    { qb_free(p1_n);    }
    if (NULL != confi_n) { qb_free(confi_n); }
    return QB_E_MALLOC;
  }

  // normalise everything, scale to u8_t:
  for (sn=0; sn < srclen_smps; sn++) {
  
    float p0, p1, confi;

    p0 = goertzel[0][sn];
    p1 = goertzel[1][sn];
    
    confi = fabsf(p0 - p1);
    
    // decide how to normalise the powers:
    qb_normalise_powers (&p0,
                         &p1,
                         max_powerX);
    confi /= max_confidence;

    p0_n[sn]    = qb_float_to_s16 (p0);
    p1_n[sn]    = qb_float_to_s16 (p1);
    confi_n[sn] = qb_float_to_s16 (confi);
    
  }
  
  isf_p0 = inspect->files + QB_INSPECT_FILE_IX_P0;
  isf_p1 = inspect->files + QB_INSPECT_FILE_IX_P1;
  isf_c  = inspect->files + QB_INSPECT_FILE_IX_CONFIDENCE;

  e1 = qb_inspect_append_s16_buf (isf_p0, p0_n,    srclen_smps);
  e2 = qb_inspect_append_s16_buf (isf_p1, p1_n,    srclen_smps);
  e3 = qb_inspect_append_s16_buf (isf_c,  confi_n, srclen_smps);
    
  // free data buffers
  qb_free(p0_n);
  qb_free(p1_n);
  qb_free(confi_n);
  
  if (QB_E_OK != e1) { return e1; }
  if (QB_E_OK != e2) { return e2; }

  return e3;

}



qb_err_t qb_inspect_write_features (qb_inspect_t *inspect,
                                    qb_wave_feature_t *feats,
                                    s64_t num_feats,
                                    s64_t input_len) {

  s16_t *buf, *p;
  s64_t smpnum, fn;
  s64_t alloc;
  qb_err_t e;
  s64_t len_acc;
  
  alloc  = 0;
  smpnum = -1;
  buf    = NULL;
  len_acc = 0;
  
  e = QB_E_OK;
  
  for (fn=0; fn < num_feats; fn++) {
  
    qb_wave_feature_t *feat;
    s64_t len;
    
    feat = feats + fn;
    len = feat->sample_num - smpnum;
    
//printf("%lld\n", feat->sample_num);
//if (fn > 10000) { exit(0); }
    
    // sanity (also potential size_t / s64 disaster)
    if (len > 0xfffffff) {
      fprintf(QB_ERR, "B: %s: [%lld] Deranged feature gap length (%lld)\n", QB_FUNC_M, smpnum, len);
      e = QB_E_BUG; // probably?
      break;
    }
    
    if (alloc < len) {
      p = qb_realloc (buf, sizeof(s16_t) * (size_t) len);
      if (NULL == p) {
        fprintf(QB_ERR, "E: Out of memory allocating inspect features write buffer.\n");
        e = QB_E_MALLOC;
        break;
      }
      buf = p;
    }
    
    memset (buf, 0, sizeof(s16_t) * len);
    
    buf[len-1] = (feat->value == QB_SYNC_WALK_FEATURE_MAXIMUM) ? 20000 : -20000;
    
    e = qb_inspect_append_s16_buf (inspect->files + QB_INSPECT_FILE_IX_WALK_FEATURES, buf, len);
    if (QB_E_OK != e) { break; }
    len_acc += len;
    
    smpnum = feat->sample_num;
    
  }
  
  if (NULL != buf) {
    qb_free(buf);
    buf = NULL;
  }
  
  // leftover piece
  input_len -= len_acc;
  if (input_len > 0) {
    buf = qb_malloc(input_len * sizeof(s16_t));
    if (NULL == buf) {
      fprintf(QB_ERR, "E: Out of memory allocating inspect features write buffer (final piece).\n");
      e = QB_E_MALLOC;
    } else {
      memset(buf, 0, input_len * sizeof(s16_t));
      e = qb_inspect_append_s16_buf (inspect->files + QB_INSPECT_FILE_IX_WALK_FEATURES, buf, input_len);
      qb_free(buf);
      buf = NULL;
    }
  }
  
  return e;
  
}


#ifdef QB_EXPERIMENTAL_INSPECT_WRITE_FEATURES_TIMING
qb_err_t qb_inspect_write_features_timing  (qb_inspect_t *inspect,
                                            qb_wave_feature_t *feats,
                                            s64_t num_feats,
                                            s64_t input_len,
                                            s32_t rate) {
                                            
  s16_t *timing_buf;
  s64_t fn;
  s64_t prev_sn;
  s64_t three_quarters;
  s64_t nominal_gap;
  qb_err_t e;
  
  timing_buf = qb_malloc (sizeof(s16_t) * input_len);
  if (NULL == timing_buf) {
    fprintf(QB_ERR, "E: Out of memory allocating features timing inspect buffer.\n");
    return QB_E_MALLOC;
  }
  memset(timing_buf, 0, sizeof(s16_t) * input_len);
  
  nominal_gap = (rate / (2 * QB_FREQ_1));
  
  prev_sn = 0;
  three_quarters = (3 * nominal_gap) / 4;
  
  for (fn=0; fn < (num_feats - 1); fn++) {
    qb_wave_feature_t *f0, *f1;
    s64_t n;
    s64_t gap;
    s16_t v;
    double delta;
    f0 = feats + fn;
    f1 = f0 + 1;
    gap = f1->sample_num - f0->sample_num;
#ifdef QB_SANITY
    if (gap < 1) {
      fprintf(QB_ERR, "B: feature gap < 1\n");
      qb_free(timing_buf);
      return QB_E_BUG;
    }
#endif
    // nominally, gap should be either nominal_gap for 1200 or (0.5 * nominal_gap) for 2400
    // if it's less than three-quarters, we'll assume it's the shorter one, the 2400
    // and double it, so that the output trace shows a consistent 1200 value for
    // either of the two nominal gap lengths (1200 or 2400)
    if (gap <= three_quarters) {
      gap *= 2;
    }
//printf("gap - nominal_gap = %lld\n", gap - nominal_gap);
    // now we'll get the difference between the measured value and the nominal one,
    // and scale it
    // ( not sure what we should scale it over )
    delta = ((double) (gap - nominal_gap)) / (3.0 * ((double) nominal_gap));
    // clamp it
    if (delta < -1.0) {
      delta = -1.0;
    } else if (delta > 1.0) {
      delta = 1.0;
    }
    v = qb_double_to_s16(delta);
    for (n=0; n < (f0->sample_num - prev_sn); n++) {
      timing_buf[f0->sample_num + n] = v;
    }
    prev_sn = f0->sample_num;
  }
  e = qb_inspect_append_s16_buf(inspect->files + QB_INSPECT_FILE_IX_WALK_FEATURES_TIMING,
                                timing_buf,
                                input_len);
  qb_free(timing_buf);
  return e;
}
#endif // QB_EXPERIMENTAL_INSPECT_WRITE_FEATURES_TIMING


qb_err_t qb_inspect_append_bitsamples (qb_span_t *span0,
                                       qb_span_t *span1,
                                       s64_t *prev_final_smpnum_inout,
                                       s32_t span0_ix,
                                       qb_inspect_file_t *f) {

  s64_t j;
  s32_t i;
  s16_t *buf;
  qb_err_t e;
  s64_t pos;
  s64_t limit, len;
  qb_atom_t *cycs;
  
  cycs = span0->atoms;
  
  /*
  if (NULL == span1) {
    // final span -- no subsequent span
    if (cycs != NULL) {
      // use the final cycle of this span as end limit instead
      limit = cycs[span0->num_atoms].sample_num;
    } else {
      // this span has no cycles yet; use rough span->start + span->len figure instead
      limit = span0->start + span0->len;
    }
  } else {
    // we have a span1, so use either its first cycle, or its rough start time, as the limit
    limit = qb_get_span_start_accurate_or_rough(span1);
  }
  */
  
  if ((cycs == NULL) || (QB_SPAN_TYPE_SILENT == span0->type)) {
    // use rough end-of-span as limit
    limit = span0->start + span0->len;
  } else {
    // have cycles => use final cycle in this span as limit
    limit = cycs[span0->num_atoms - 1].sample_num;
  }
  
//printf("span %d: limit = %lld\n", span0_ix, limit);
  
  len = limit - *prev_final_smpnum_inout; //qb_get_span_start_accurate_or_rough(span0);
  
  // sanity
  if ((len < 1) || (len > 0xffffffff)) {
    fprintf(QB_ERR, "B: %s: stupid allocation (%lld x 2 bytes)\n", QB_FUNC_M, len);
    return QB_E_BUG;
  }
  
  // allocate enough for span0
  buf = qb_malloc(sizeof(s16_t) * len);
  
  if (NULL == buf) {
    fprintf(QB_ERR, "E: Out of memory writing halfbit instants inspection file.\n");
    return QB_E_MALLOC;
  }
  
  memset(buf, 0, sizeof(s16_t) * len);
  
  e   = QB_E_OK;
  pos = len; // default, if no cycles available
  
  if (span0->atoms != NULL) {
  
    // have some cycles
  
    pos = 0;
    for (i=0; (i < span0->num_atoms) && (pos < len); i++) {
    
      // samples_since_last may not be set on cycles yet,
      // so we can't use that
      
      s64_t smps_since_last;
      s8_t flip;
      
//printf("cyc sample_num = %lld\n", cycs[i].sample_num);
      
      smps_since_last = cycs[i].sample_num - *prev_final_smpnum_inout;
      
      // hmm ...
      if (0 == smps_since_last) {
        continue;
      }
      
      if (    (smps_since_last > HACK_DBG_WRITE_BITSAMPLES)
           || (smps_since_last < 1) ) {
        fprintf(QB_ERR,
                "B: %s: [%lld]: span %d/%d, type %s/%s, cyc %d, prev_smpnum %lld: smps_since_last is bad (%lld)\n",
                QB_FUNC_M,
                cycs[i].sample_num,
                span0_ix,
                (NULL!=span1)?(span0_ix+1):0,
                qb_span_get_type_text(span0),
                (NULL!=span1)?qb_span_get_type_text(span1):"null",
                i,
                *prev_final_smpnum_inout,
                smps_since_last);
        qb_free(buf);
        return QB_E_BUG;
      }
      
      if (pos < len) {
        // flip over cycles that are either synthetic (walk interpolation)
        // or inserted (PLL odd-run error check)
        if ((i>0) && (cycs[i-1].inserted || cycs[i-1].synthetic)) {
          flip = -1;
        } else {
          flip = 1;
        }
        buf[pos] = 20000 * flip;
        pos++;
      }

      for (j=0; (j < (smps_since_last - 1)) && (pos < len); j++) {
        buf[pos] = 2000;
        pos++;
      }
      
      *prev_final_smpnum_inout = cycs[i].sample_num;
      
    }
    
  } // endif (span has cycles)
  
//if (QB_SPAN_TYPE_DATA == span0->type) {
//  exit(0);
//}
  
  /*
  if (pos > len) {
    fprintf(QB_ERR, "E: qb_inspect_append_bitsamples: somehow cycle %d "
                    "implies a smpnum > orig. input len (%lld vs %lld)\n",
                    i, pos, len);
    e = QB_E_BUG;
  } else {
    e = qb_inspect_append_s16_buf (f, buf, pos);
  }
  */
  
  e = qb_inspect_append_s16_buf (f, buf, len);
  *prev_final_smpnum_inout = limit; // update start limit for next span
  
//printf("span %d: prev_final_smpnum_inout = %lld at return\n", span0_ix, *prev_final_smpnum_inout);
  
  qb_free(buf);

  return e;
  
}


qb_err_t qb_inspect_append_s16_buf (qb_inspect_file_t *ifile,
                                    s16_t *buf,
                                    s64_t len) {
  sf_count_t c;
  qb_err_t e;
  e = qb_inspect_open_if_needed(ifile);
  if (QB_E_OK != e) { return e; }
  c = sf_write_short (ifile->sf, buf, len);
  if (c != len) {
    fprintf(QB_ERR, "E: Failed to append to inspect file: %s\n", ifile->priv.path);
    return QB_E_SNDFILE_WRITE;
  }
  return QB_E_OK;
}



qb_err_t qb_inspect_open_if_needed (qb_inspect_file_t *ifile) {

  qb_stat_t qbstat;
  qb_err_t e;
  qb_libsndfile_inspect_privdata_t *sfd;
  FILE *f;
  char *path_buf;
  
//printf("qb_inspect_open_if_needed\n");
  
  sfd = &(ifile->priv);
  
  if ( sfd->opened ) { return QB_E_OK; }
  
  e = qb_inspect_build_and_stat_path (ifile->dir, ifile->filename, &path_buf);
  if (QB_E_OK != e) { return e; }
  
  memset(&(ifile->priv), 0, sizeof(qb_libsndfile_inspect_privdata_t));
  ifile->priv.path = path_buf;

  memset(&qbstat, 0, sizeof(qb_stat_t));
  
  do {
  
    e = qb_stat_file (sfd->path, &qbstat);
    if ((QB_E_OK != e) && (QB_E_STAT != e)) {
      break;
    }
    
    if (QB_E_OK == e) { // file exists
      if ( ! ifile->allow_overwrite ) { // see if it may be overwritten
        fprintf(QB_ERR, "E: Refusing to overwrite file: %s\n", sfd->path);
        e = QB_E_FOPEN_OPFILE;
        break;
      }
      if (qbstat.is_dir || qbstat.is_link) {
        fprintf(QB_ERR, "E: Cannot overwrite inspect file, as it is not a file: %s\n", sfd->path);
        e = QB_E_FOPEN_OPFILE;
        break;
      }
    }
    
    e = QB_E_OK; // catch QB_E_STAT
    
    f = qb_fopen (sfd->path, "wb");
    if (NULL == f) {
      fprintf(QB_ERR, "E: Could not open inspect file for writing: %s\n", sfd->path);
      e = QB_E_FOPEN_OPFILE;
      break;
    }

    e = qb_inspect_file_init (ifile, f);
    if (QB_E_OK != e) { break; }
    
    //if (NULL == sfd->path) {
    //  fprintf(QB_ERR, "B: sfcb_inspect_open_if_needed: sfd->path is NULL\n");
    //  return QB_E_BUG;
    //}
    
  //printf("sfcb_inspect_open_if_needed(): %s\n", ifile->priv.path);
  //exit(0);

    printf("  inspect -> %s\n", sfd->path);
    
    sfd->opened = 1;
    sfd->len = 0;
    sfd->off = 0;
  
  } while (0);
  
  if (QB_E_OK != e) {
    qb_free(ifile->priv.path);
    ifile->priv.path = NULL;
  }
  
  return e;
  
}


// callbacks for INSPECT WRITE (sfcb_inspect_...)
static sf_count_t sfcb_inspect_get_filelen (void *priv) {
//~ printf("sfcb_get_filelen()\n");

  qb_libsndfile_inspect_privdata_t *sfd;
  
  sfd = (qb_libsndfile_inspect_privdata_t *) priv;
  
  //e = sfcb_inspect_open_if_needed (sfd);
  //if (QB_E_OK != e) { return -1; }
  
  return sfd->len;
  
}


static sf_count_t sfcb_inspect_seek (sf_count_t offset, int whence, void *priv) {

//~ printf("sfcb_seek(off=%lld, whence=%d)\n", (s64_t) offset, whence);

  qb_libsndfile_inspect_privdata_t *sfd;
  sf_count_t pos;
  int i;
  
  sfd = (qb_libsndfile_inspect_privdata_t *) priv;
  
  if (NULL == sfd->f) {
    fprintf(QB_ERR, "B: %s: priv->f is NULL\n", QB_FUNC_M);
    return -1;
  }
  
  //e = qb_inspect_open_if_needed (sfd);
  //if (QB_E_OK != e) { return -1; }
  
  pos = sfd->off;
  
  if (SEEK_SET == whence) {
    pos = 0;
  } else if (SEEK_END == whence) {
    pos = sfd->len;
  }
  
  pos += offset;
  
  if (pos < 0) {
    // do something
    pos = 0;
  } else if (pos > sfd->len) { // it's allowed to be one past the end (EOF)
    // do something
    // is this an error or not?
    // just set it one past the end
    pos = sfd->len;
  }
 
#ifdef QB_WINDOWS
  i = _fseeki64(sfd->f, pos, SEEK_SET);
#else
  i = fseeko (sfd->f, pos, SEEK_SET);
#endif
  
  if (0 != i) {
    fprintf(QB_ERR, "E: inspect file seek to %lld failed.\n", (s64_t) pos);
    return -1;
  }
  
  sfd->off = pos;
  
  return pos;
  
}

static sf_count_t sfcb_inspect_read (void *q, sf_count_t count, void *priv) {
  // shut the stupid compiler up:
  void *p;
  sf_count_t lol;
  p = q;
  q = p;
  p = priv;
  priv = p;
  lol = count;
  count = lol;
  fprintf(QB_ERR, "B: libsndfile called %s -- shouldn't happen\n", QB_FUNC_M);
  return -1;
}

// not actually needed, libsndfile shouldn't call it:
/*
static sf_count_t sfcb_inspect_read (void *q, sf_count_t count, void *priv) {

  qb_libsndfile_inspect_privdata_t *sfd;
  sf_count_t wanted;
  qb_err_t e;
  
  sfd = (qb_libsndfile_inspect_privdata_t *) priv;
  
  e = sfcb_inspect_open_if_needed (sfd);
  if (QB_E_OK != e) { return -1; }

//~ printf("%lu\n", sizeof(sf_count_t)); exit(0);
//~ printf("sfd.buf = %p\n", sfd->buf);
//~ printf("read pos is %lld\n", (s64_t) sfd->pos);
//~ printf("sfcb_read(%lld) { ", (s64_t) count);
  wanted = count;
  if ((sfd->off + wanted) > sfd->len) {
    wanted = sfd->len - sfd->off; // truncate
  }
#ifdef QB_SANITY
  if (wanted < 0) {
    fprintf(QB_ERR,
            "\nsfcb_alpha2_read: insane libsndfile somehow requested negative num bytes!\n");
    return 0;
  }
#endif

  e = qb_fread_fully (q, wanted, sfd->f);
  if (QB_E_OK != e) { return -1; }
  
  sfd->off += wanted;
  
  return wanted;
  
}
*/

#define QB_SNDFILE_WRITE_ALLOC_DELTA 1000000

static sf_count_t sfcb_inspect_write (const void *q, sf_count_t count, void *priv) {
  
  qb_libsndfile_inspect_privdata_t *ptr;
  //u8_t *buf;
  //s64_t new_alloc;
  u8_t *q8;
  qb_err_t e;
  
  q8 = (u8_t *) q;
  
  ptr = (qb_libsndfile_inspect_privdata_t *) priv;
  
//printf("sfcb_inspect_write: %zu bytes, file %s\n", (size_t) count, ptr->path);
  
  //e = qb_sfcb_inspect_open_if_needed (ptr);
  //if (QB_E_OK != e) { return -1; }
  
#ifdef QB_SANITY
  if (count > QB_MAX_IPFILE_SAMPLES) {
    fprintf(QB_ERR,
            "\nsfcb_alpha2_write: insane libsndfile somehow tried to write %lld bytes\n",
            (s64_t) count);
    return -1;
  }
  if ((ptr->off + count) > QB_MAX_IPFILE_SAMPLES) {
    fprintf(QB_ERR,
            "\nsfcb_alpha2_write: tried to write more total audio frames than are allowed (%lld vs. %lld)\n",
            (s64_t) (ptr->len + count),
            QB_MAX_IPFILE_SAMPLES);
    return -1;
  }
#endif
  
  e = qb_fwrite_fully (q8, count, ptr->path, ptr->f);
  if (QB_E_OK != e) { return -1; }
  
  (ptr->off) += count;
  
  if (ptr->off >= ptr->len) {
    ptr->len = 1 + ptr->off;
  }
  
  return count;
  
}



static sf_count_t sfcb_inspect_tell (void *priv) {
//~ printf("sfcb_tell()\n");
  qb_libsndfile_inspect_privdata_t *sfd;
  sfd = (qb_libsndfile_inspect_privdata_t *) priv;
  return sfd->off;
}


void qb_close_inspect_file (qb_inspect_file_t *inspect_file) {

  qb_libsndfile_inspect_privdata_t *priv;
  
  if (NULL != inspect_file->sf) {
    sf_close(inspect_file->sf);
  }
  
  priv = &(inspect_file->priv);

  if (NULL != priv->f) {
    fclose(priv->f);
  }
  if (NULL != priv->path) {
    qb_free(priv->path);
  }
  
  memset(priv, 0, sizeof(qb_libsndfile_inspect_privdata_t));
  
}
