/*
 *  Quadbike 2
 *  Copyright (C) 2026 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "atom.h"
#include "source.h"
#include "util.h"
#include "errcorr.h"
#include "span.h"

#include <string.h>
#include <stdio.h>
#include <math.h>
#include <stdlib.h>

static qb_err_t squawk_strip_rogue_leading_1200_cycs (qb_span_t *span);


// I think this was done because sometimes leading squawk 0s get interpreted as start bits?
// I remember the emulators occasionally seeing a rogue block that wasn't actually there,
// I think this is why this code was introduced? should have kept a note of which source it was :(
static qb_err_t squawk_strip_rogue_leading_1200_cycs (qb_span_t *span) {

  s32_t an;
  
  an=0;
  
//printf("squawk_strip_rogue_leading_1200_cycs: span->start = %lld, type \"%s\"\n", span->start, qb_span_get_type_text(span));

  if (0 == span->num_atoms) {
    fprintf(QB_ERR, "\nB: %s: no cycles?!\n", QB_FUNC_M);
    return QB_E_BUG;
  }
//printf("%c%c%c\n",span->atoms[0].value,span->atoms[1].value,span->atoms[2].value);
  while ((an < span->num_atoms) && ('0' == span->atoms[an].value)) {
    an++;
  }
  if (an == span->num_atoms) {
    fprintf(QB_ERR,
            "\nW: [%lld] squawk_strip_rogue_leading_1200_cycs: all cycles were '0'?!\n",
            span->atoms[0].sample_num);
    // do nothing?
    return QB_E_SQUAWK_ALL_ZEROS;
  }
  span->num_atoms -= an;
  memmove (span->atoms, span->atoms + an, sizeof(qb_atom_t) * span->num_atoms);
  return QB_E_OK;
}

void qb_init_atom (qb_atom_t *c) {
  memset(c, 0, sizeof(qb_atom_t));
  //c->orig_sample_num = -1;
  c->pll_lock = 1.0;
  //~ c->is_valid = 0;
}



qb_err_t qb_span_measure_atom_powers (float *goertz[2],
                                      float goertz_hf_test_multiplier, // 2.0.4, for scan only (else 1.0)
#ifdef QB_SANITY
                                       s64_t goertzel_len_smps,
#endif
                                       // gets updated: timing in, power out:
                                       qb_atom_t *atoms,
                                       s32_t num_atoms,
                                       u8_t update_atoms, // set to 1 once optimum shift established, to update atom timings
                                       s8_t shift_smps,
                                       float *summed_confidence_out) { // or NULL, if update_atoms is 0

  s32_t q;
  qb_err_t e;
  float hf; // 2.0.4

  e = QB_E_OK;
  
  if ((NULL != summed_confidence_out) && update_atoms) {
    fprintf(QB_ERR, "B: %s: non-NULL summed_confidence_out but update_atoms is 1\n", QB_FUNC_M);
    return QB_E_BUG;
  }
  
  if (NULL != summed_confidence_out) {
    *summed_confidence_out = 0.0f;
  }
  
  // loop per-atom
  for (q=0; q < num_atoms; q++) {
  
    s64_t sn;
    qb_atom_t *atom;
    float confidence;
    
    atom = atoms + q;

#ifdef QB_SANITY
    if (atom->sample_num < 0) {
      // memory corruption
      fprintf(QB_ERR, "B: %s: atoms list element %d uninitialised??\n",
              QB_FUNC_M, q);
      e = QB_E_BUG;
      break;
    }
#endif
    
    // get sample number from atom instant
    sn = atom->sample_num + shift_smps;

#ifdef QB_SANITY
    if (sn >= goertzel_len_smps) {
      fprintf(QB_ERR,
              "B: %s: atom %d sample number (%lld) exceeds length of Goertzel data (%lld)!\n",
              QB_FUNC_M, q, sn, goertzel_len_smps);
      e = QB_E_BUG;
      break;
    }
#endif

    //if (update_cycs) {
      // rewrite cycle power
    //  cyc->sample_num = sn;
    //}

    hf = goertz_hf_test_multiplier * goertz[1][sn]; // 2.0.4: multiplier
    
    confidence = fabsf(goertz[0][sn] - hf);

    // store everything on atoms[]
    if (update_atoms) {
      atoms[q].power0 = goertz[0][sn];
      atoms[q].power1 = hf; // 2.0.4
      atoms[q].confidence = confidence;
    }
    
    if (NULL != summed_confidence_out) {
      *summed_confidence_out += confidence;
    }
    
    
  } // next atom
  
  return e;
  
}

#define QB_SPAN_SHIFTS_PRINTED_PER_LINE 18

qb_err_t qb_take_power_samples (
#ifdef QB_SANITY
                                s64_t srclen_smps,
#endif
                                qb_span_t *spans_inout,
                                s32_t num_spans,
                                u8_t suppress_output,
                                //u8_t do_shift_search,
                                u8_t use_forced_shift,
                                s8_t forced_shift,
                                float *goertz[2],
                                float goertz_hf_test_multiplier) { // 2.0.4 -- for scanning, 1.0 elsewhere

  qb_err_t e;
  s32_t sn;
  s8_t shift_low, shift_high;
  s32_t spans_done;
  s8_t last_span_best_shift;
  s32_t last_data_span_ix; // tedious, rancid cosmetics
  
  e = QB_E_OK;
  
  // find last data span because of rotting, putrefying cosmetics
  last_data_span_ix = 0;
  for (sn = num_spans-1; sn >= 0; sn--) {
    if (QB_SPAN_TYPE_DATA == spans_inout[sn].type) {
      last_data_span_ix = sn;
      break;
    }
  }
  
  if ( ! use_forced_shift ) {
    shift_low  = -QB_SPAN_SHIFT_SEARCH_HALF_RANGE;
    shift_high = QB_SPAN_SHIFT_SEARCH_HALF_RANGE;
    if (!suppress_output) {
      printf("    Data span optimum shift search (+/- %d samples):\n      ", QB_SPAN_SHIFT_SEARCH_HALF_RANGE);
    }
  } else if (!suppress_output) {
    printf("    Using fixed shift value of %d samples.\n", forced_shift);
  }// else {
   // printf("    Using shift value of 0.\n");
 // }
  
  spans_done = 0;
  last_span_best_shift = 127; // something invalid
  
  for (sn=0; (QB_E_OK == e) && (sn < num_spans); sn++) {
  
    s8_t shift;
    float best_confidence;
    s8_t best_shift;
    u8_t squawk_char;
    qb_span_t *span;
    
    span = spans_inout + sn;
    
    best_shift = 0;
    best_confidence = 0.0f;
    
    if (QB_SPAN_TYPE_DATA != span->type) {
      continue; // data spans only
    }
    
    if ( ! use_forced_shift ) {
    
      shift_low  = -QB_SPAN_SHIFT_SEARCH_HALF_RANGE;
      shift_high =  QB_SPAN_SHIFT_SEARCH_HALF_RANGE;
    
      for (shift = shift_low; shift <= shift_high; shift++) {
      
        float summed_confidence;
        
        summed_confidence = 0.0f;
        
        e = qb_span_measure_atom_powers (goertz,
                                         goertz_hf_test_multiplier,
#ifdef QB_SANITY
                                          srclen_smps,
#endif
                                          span->atoms,
                                          span->num_atoms,
                                          0, // don't update atom powers yet
                                          shift,
                                          &summed_confidence);
        if (QB_E_OK != e) { break; }
        
//printf("span %u shift %d summed confidence %f\n", sn, shift, summed_confidence);
        
        if (summed_confidence > best_confidence) {
          best_confidence = summed_confidence;
          best_shift = shift;
        }
        
      } // next shift value
      
//printf("span %u BEST SHIFT %d\n", sn, best_shift);
      
    } else {
      best_shift = forced_shift;
    }
    
    if (QB_E_OK != e) { break; }
    
    if ( ! use_forced_shift ) {
      squawk_char = qb_is_squawk (span, (sn>0)?(span-1):NULL) ? 's' : ' ';
      if (last_span_best_shift != best_shift) {
        printf ("%+d%c ", best_shift, squawk_char);
      } else {
        printf ("..%c ", squawk_char);
      }
      if ((spans_done % QB_SPAN_SHIFTS_PRINTED_PER_LINE) == (QB_SPAN_SHIFTS_PRINTED_PER_LINE-1)) {
        printf("\n");
        if (sn != last_data_span_ix) {
          printf ("      "); // indent for next line, but not if final line
        }
      }
      last_span_best_shift = best_shift;
    }
    
    // now actually update the timing on the atoms with best shift value
    // (or zero, or provided shift value)
    e = qb_span_measure_atom_powers  (goertz,
                                      goertz_hf_test_multiplier,
#ifdef QB_SANITY
                                      srclen_smps,
#endif
                                      span->atoms,
                                      span->num_atoms,
                                      1, // OK, update atom powers now
                                      best_shift,
                                      NULL);
    if (QB_E_OK != e) { break; }
    
    if ( ! use_forced_shift ) {
      spans_done++;
    }
                                     
  } // next span
  
  if ( ( ! use_forced_shift ) && ((spans_done % QB_SPAN_SHIFTS_PRINTED_PER_LINE) != 0) ) {
    printf("\n");
  }

  return e;

}



qb_err_t qb_atoms_assign_values (qb_source_t *src,
                                     u8_t dp) { // display_progress

  qb_err_t e;

  e = QB_E_OK;
  
  do {
  
    qb_span_t *spans;
    s32_t num_spans;
    s32_t q;
    s32_t sn;
    s32_t num_transient_leader_zeros;

    //printf("Source #%u:\n", j+1);

    // loop per source
    num_transient_leader_zeros = 0;
    
    spans = src->spans;
    num_spans = src->num_spans;
    
    if (NULL == spans) {
      //printf("  source disabled\n");
      //continue;
      fprintf(QB_ERR, "\nB: %s: spans is NULL\n", QB_FUNC_M);
      //return QB_E_NO_SOURCES;
      return QB_E_BUG;
    }
    
    printf("  Measuring Goertzel power at half-bit intervals: ");
    fflush(stdout);
    qb_show_meter(dp);

    for (sn=0; sn < num_spans; sn++) {
    
      qb_span_t *span, *prev;
      
      span = spans + sn;
      
      if (QB_SPAN_TYPE_SILENT == span->type) {
        qb_update_meter (dp, sn, num_spans, 1.0f, 0);
        continue;
      }
    
      for (q=0; q < span->num_atoms; q++) {
        span->atoms[q].value = qb_atom_get_value_bool (span->atoms + q) ? '1' : '0';
      } // next atom
      
      // strip any rogue 1200 Hz atoms from the start of squawk spans.
      // (they always start with 2400-Hz cycles).
      prev = (sn>0) ? (span-1) : NULL;
      if ( qb_is_squawk (span, prev) ) {
        e = squawk_strip_rogue_leading_1200_cycs (span);
        if (QB_E_SQUAWK_ALL_ZEROS == e) {
          // no idea what we do about this
          // how about nothing?
          qb_show_meter(dp); // re-establish this
          e = QB_E_OK;
        }
      }
      if (QB_E_OK != e) { break; }
      
      qb_update_meter (dp, sn, num_spans, 1.0f, 0);
      
    } // next span
    
    if (QB_E_OK == e) {
      qb_hide_meter (dp, 0);
      printf("done.\n");
    } else {
      break;
    }
    
    if (num_transient_leader_zeros) {
      fprintf(QB_ERR, "  W: ignored %d transient leader zeros.\n",
              num_transient_leader_zeros);
    }
    
    if (QB_E_OK != e) { break; }

  } while (0);

  return e;
  
}




#define QB_CYCLES_ALLOC_DELTA 100000

qb_err_t qb_append_atom  (qb_atom_t *si,
                          qb_atom_t **atoms_inout,
                          s32_t *alloc,
                          s32_t *fill ) {
                           
  qb_atom_t *atoms_new;

  if (*fill >= QB_MAX_ATOMS) {
    fprintf(QB_ERR,
            "\nE: Maximum number of atoms exceeded (%d).\n",
            QB_MAX_ATOMS);
    return QB_E_TOO_MANY_CYCLES;
  }
  if (*fill < 0) {
    fprintf(QB_ERR, "\nB: %s: fill < 0 (%d)\n", QB_FUNC_M, *fill);
    return QB_E_BUG;
  }

  if ( ( *fill + 1 ) >= *alloc ) {
    *alloc += 1 + QB_CYCLES_ALLOC_DELTA;
    atoms_new = qb_realloc (*atoms_inout,
                            sizeof(qb_atom_t) * *alloc);
    if (NULL == atoms_new) {
      fprintf(QB_ERR, "\nE: Out of memory appending new atom.\n");
      return QB_E_MALLOC;
    }
    *atoms_inout = atoms_new;
  }
  (*atoms_inout)[*fill] = *si;
  (*fill)++;
  
  return QB_E_OK;
  
}


/*
qb_err_t qb_cycles_check_integrity (qb_atom_t *cycs, s32_t num_cycs) {

  // for development -- check that samples_since_last and sample_num
  // are consistent for every cycle in the pack
  
  s32_t n;
  s64_t a; // the accumulator
  
  a = -1;
  
  for (n=0; n < num_cycs; n++) {
    qb_atom_t *c;
    c = cycs + n;
    a += c->samples_since_last;
    if (a != c->sample_num) {
      fprintf(QB_ERR, "B: Cycs. integrity check: cyc %d: acc != c->smpnum (%lld vs %lld)\n",
              n, a, c->sample_num);
      return QB_E_BUG;
    }
  }
  
  printf("Cycle integrity check passed.\n");
  
  return QB_E_OK;
  
}
*/


qb_err_t qb_fill_missing_atoms_on_leader_spans_and_adjust_limits (qb_span_t *spans,
                                                                  s32_t num_spans,
                                                                  s32_t rate,
                                                                  s64_t total_input_smps) {
                                                          
  /*
  
               |               |
    D..0 D..1 D..2 L..3 L..4 L..5 D..6 D..7 D..8  < atoms
    ^    ^    ^|   ^    ^    ^ |  ^    ^    ^     < atom start times; atom->sample_num
               |               |                  < spans; we arrange the spans so that the START of the leader span
      DSPAN    |     LSPAN     |      DSPAN                happens DURING the final atom of the data span,
               |               |                           (just as long as the final data atom STARTS during the data span).
               |               |                           but the end of it protrudes into the leader span.
              |                   |               < adjusting the span's start/end times based on atom->sample_num
              |                   |                 for the final previous data atom and initial subsequent data atom
              |                   |                 gives us this
                   |              |               < then we move the start forward by one atom (based on measured tapespeed);
                   |              |                 this is the piece we need to divide into leader atoms.

  */
                                                          
  qb_err_t e;
  s32_t i;
  e = QB_E_OK;
  
  for (i=0; i < num_spans; i++) {
  
    s64_t start, end;
    u8_t prev_span_is_valid, prev_span_is_data;
    u8_t next_span_is_valid, next_span_is_data;
    double nom;
    double num_atoms_fractional;
    s32_t num_atoms_int;
    double atom_len_f;
    double leader_smps_accumulated;
    s32_t atom;
    s32_t atoms_alloc;
    s64_t last;
    s64_t nom_int;
    
    prev_span_is_data = 0;
    next_span_is_data = 0;
    
    prev_span_is_valid = (i>0);
    next_span_is_valid = (i < (num_spans - 1));
    
    if (prev_span_is_valid) {
      prev_span_is_data = (QB_SPAN_TYPE_DATA == spans[i-1].type);
    }
    if (next_span_is_valid) {
      next_span_is_data = (QB_SPAN_TYPE_DATA == spans[i+1].type);
    }
    
    // defaults
    start = 0;
    //end = spans[num_spans-1].start + spans[num_spans-1].len; // end of input
    end = total_input_smps;
    

//printf("spans[%d].num_atoms = %d\n", i, spans[i].num_atoms);

/*
if (prev_span_is_valid && prev_span_is_data) {
  printf("prev span cycle smps range %lld -> %lld\n",
         spans[i-1].atoms[0].sample_num,
         spans[i-1].atoms[spans[i-1].num_atoms - 1].sample_num);
}
*/

    // we already have cycles for data spans,
    // so we're interested in non-data spans only
    if (QB_SPAN_TYPE_LEADER != spans[i].type) {
      continue;
    }
    
  
    if (prev_span_is_valid) {
      // note that a data span may have NULL cycs if the PLL didn't
      // get an opportunity to lock (i.e. the initial AA 'squawk'):
      if ( ( ! prev_span_is_data ) || (NULL == spans[i-1].atoms) || (0 == spans[i-1].num_atoms)) { // PREVIOUS
        // by default, use the span's time range
        start = spans[i].start;
      } else {
        // but, if previous span is a data span, use its final cycle's time instead
        start = spans[i-1].atoms[spans[i-1].num_atoms - 1].sample_num;
        // need to add on one cycle period
        //start += (s64_t) round (rate / (QB_FREQ_2 * spans[i-1].speed));
        start += (s64_t) roundf (rate / (((float) QB_FREQ_2) * spans[i-1].speed));
      }
    }
    
    if (next_span_is_valid) {
      if ( ! next_span_is_data || (NULL == spans[i+1].atoms) ) { // NEXT
        // by default, use the span's time range
        end = spans[i].start + spans[i].len;
      } else {
        // but, if next span is a data span, use its first cycle's time instead
        end = spans[i+1].atoms[0].sample_num;
#ifdef QB_SANITY
        if ((end - (spans[i].start + spans[i].len)) < 0) {
          fprintf(QB_ERR,
                  "W: [%lld] Data cycle protrudes into previous leader span (#%d) by %lld smps.\n",
                  end, i, (spans[i].start + spans[i].len) - end);
          // FIXME: what do we do about this?
        }
#endif
        //printf("using end based on cycles: %lld (not %lld, delta %lld)\n",
        //       end,
        //       spans[i].start + spans[i].len,
        //       end - (spans[i].start + spans[i].len));
      }
    }
    
    nom = (((double) rate) / (QB_FREQ_2 * spans[i].speed));
    
    num_atoms_fractional = ((double) (end - start)) / nom; // a fraction
    num_atoms_int        = (s32_t) round(num_atoms_fractional);  // an integer
    atom_len_f           = ((double) (end - start)) / (double) num_atoms_int; // revised cycle len, a fraction
    
//printf("span %d: speed %f, nom %lf, cyc_len_f = %lf\n", i, spans[i].speed, nom, cyc_len_f);
    
    leader_smps_accumulated = 0.0;
    atoms_alloc = 0;
    
    last=0;
    nom_int = (s64_t) round(nom);
    
    for (atom=0; atom < num_atoms_int; atom++) {
    
      s64_t t;
      qb_atom_t c;
      
      t = (s64_t) round (((double)start) + leader_smps_accumulated);
      
      // make sure samples since last is sane
      if (    (atom>0)
           && (    ((t - last) > (nom_int + 1))
                || ((t - last) < (nom_int - 1)))) {
        fprintf(QB_ERR, "W: leader span %d (%lld smps): floating-point accuracy lost? SSL = %lld, nominal = %lld\n",
                i, start, (t - last), nom_int);
      }
      last = t;
      
      qb_init_atom(&c);
      c.value = 'L';
      c.sample_num = t;
      c.confidence = 10.0f; // shrug
      c.power0 = 0.0f;
      c.power1 = 10.0f; // shrug
      e = qb_append_atom (&c, &(spans[i].atoms), &atoms_alloc, &(spans[i].num_atoms));
      if (QB_E_OK != e) { return e; }
      leader_smps_accumulated += atom_len_f;
    }
    
  } // next (leader) span
  
  return e;
  
}


qb_err_t qb_create_two_atoms_on_silent_spans (qb_span_t *spans, s32_t num_spans, u8_t verbose) {
  
  // establish cycles within silent spans
  
  s32_t sn;
  
  for (sn=0; sn < num_spans; sn++) {
  
    qb_span_t *span;
    qb_atom_t c;
    s32_t alloc;
    qb_err_t e;
    
    alloc = 0;
    span = spans + sn;
    
#ifdef QB_SANITY
    if ( (NULL == span->atoms) && (0 != span->num_atoms) ) {
      fprintf(QB_ERR, "W: what?! span %d type %s (start %lld) has %d atoms but a NULL atoms field\n",
              sn, qb_span_get_type_text(span), span->start, span->num_atoms);
      
    }
#endif

    // it is possible at this time for a data
    // span not to contain any cycles (no chance
    // for the PLL to lock on ...)
    // so we'll turn them into silent cycles if this happens
    
    // note that this will necessitate another call to amalgamate_spans()
    // once this function returns
    if (    (QB_SPAN_TYPE_SILENT == span->type)
         || (NULL == span->atoms) ) {
      if (verbose && (QB_SPAN_TYPE_SILENT != span->type)) {
        printf("  W: span %d: converting %s to silent\n", sn, qb_span_get_type_text(span));
      }
      
      qb_span_set_type (span, QB_SPAN_TYPE_SILENT);
      
      qb_init_atom(&c);
      c.value = '0'; // we're abolishing silent cycles, so this is a 0-cycle.
                     // '0' because this is the single-pulse cycle type.

      // there is a potential issue with placing the
      // silent span's padding atom right at the very start of the span,
      // as we used to.
      
      // the problem is that it is possible for the previous span
      // to contain a 1-atom right at its very end. this leads to
      // a situation where we have the final data atom at sample
      // N, and then the silent atom at sample N+1. if the data
      // atom is a 1, we would need to insert a *pair* of pulses for it,
      // but we only have one sample's space in which to do that, which
      // is impossible, and leads to "zero run_len" CSW writing failure.
      
      // it's safer therefore to place the silent atom not at the very
      // start of the span, but one sample into it; so that's
      // what we're going to do.
      
      // we could check the previous atom and see where it is,
      // but I don't think there's anything to be lost by just
      // delaying the silent atom by a single sample. it
      // shouldn't make a difference to anything, as far as I
      // can see.

      c.sample_num      = span->start + 1; //hack_prevent_silent_cycle_cutting_off_final_data_cycle;
      
      c.confidence = 0.0f;
      c.power0 = 0.0f;
      c.power1 = 0.0f;
      
      e = qb_append_atom (&c, &(span->atoms), &alloc, &(span->num_atoms));
      if (QB_E_OK != e) { return e; }
      
      // second cycle
      c.sample_num = span->start + (span->len / 2);
//printf("qb_create_two_atoms_on_silent_spans: second at [%lld]\n", c.sample_num);
      e = qb_append_atom (&c, &(span->atoms), &alloc, &(span->num_atoms));
      if (QB_E_OK != e) { return e; }
      
    }// else {
    //  cycles_update_samples_since_last_for_span (span);
    //}
  }
  
  return QB_E_OK;
  
}

u8_t qb_atom_get_value_bool (qb_atom_t *atom) {
  return (atom->power1 > atom->power0);
}
