/*
 *  Quadbike 2
 *  Copyright (C) 2025 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "errcorr.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

static qb_err_t do_error_correction_for_span (qb_span_t *span,
                                              u8_t strategy,
                                              u32_t *flipped_early_out,
                                              u32_t *flipped_late_out,
                                              u32_t *flipped_0_to_1_out,
                                              u32_t *flipped_1_to_0_out,
                                              u32_t *skipped_out,
                                              u8_t verbose);
                                              
static void flip_cycle_value (qb_atom_t *c, u32_t flips[2][2], u8_t late, u8_t verbose) ;

qb_err_t qb_do_atomflip_correction_all_spans (qb_span_t *spans, s32_t num_spans, u8_t strat, u8_t verbose) {

  s32_t sn;
  qb_err_t e;
  u32_t early, late, zero_to_one, one_to_zero, skipped;
  
  if (QB_ERRCORR_NONE == strat) { return QB_E_OK; }
  
  early = late = zero_to_one = one_to_zero = skipped = 0;
  e = QB_E_OK;
  for (sn=0; sn < num_spans; sn++) {
    u32_t fe, fl, f01, f10, sk;
    qb_span_t *span;
    span = spans + sn;
    // data spans only
    if (    (QB_SPAN_TYPE_DATA != span->type)
         || qb_is_squawk (span, (sn > 0) ? (span - 1) : NULL)) { // do not correct squawks
      continue;
    }
    e = do_error_correction_for_span (span,
                                      strat,
                                      &fe,
                                      &fl,
                                      &f01,
                                      &f10,
                                      &sk,
                                      verbose);
    if (QB_E_OK != e) { return e; }
    early += fe;
    late += fl;
    zero_to_one += f01;
    one_to_zero += f10;
    skipped += sk;
  }
  printf("    Flips: early %u, late %u, 0-to-1 %u, 1-to-0 %u, skipped %u.\n",
         early,
         late,
         zero_to_one,
         one_to_zero,
         skipped);
  return e;
}


/*
//     - singletons force adjacent bitflip
//     - confidence basis
//     - force Q -> ~Q always (where Q specified as constant 0 or 1)
//     - earlier cycle always wins
//     - later cycle always wins*/

static qb_err_t do_error_correction_for_span (qb_span_t *span,
                                              u8_t strategy,
                                              u32_t *flipped_early_out,
                                              u32_t *flipped_late_out,
                                              u32_t *flipped_0_to_1_out,
                                              u32_t *flipped_1_to_0_out,
                                              u32_t *skipped_out,
                                              u8_t verbose) {

  qb_err_t e;
  s32_t cix;
  s32_t run_start_cycs;
  u8_t running_val;
  qb_cycle_run_t runs[2];
  s32_t num_cycs;
  qb_atom_t *cycs;
  
  u32_t flips[2][2]; // [early/late][false/true]
  u32_t skipped;
  
  memset(flips, 0, sizeof(u32_t) * 2 * 2);
  skipped = 0;

  e = QB_E_OK;
  
  cycs = span->atoms;
  num_cycs = span->num_atoms;
  
  // look for runs
  
  // set up "pending run"
  running_val = 0xff; // force initial update
  run_start_cycs = -1;
  
  memset(runs, 0, sizeof(qb_cycle_run_t) * 2);
  
  for (cix=0; cix < num_cycs; cix++) {
  
    qb_atom_t *c, *c0, *c1;
    u8_t value, early_value, late_value;
    u8_t zero_run_is_short;
    u8_t odd_adjacent_pair;
    
    c = cycs + cix;
    
    /*
#ifdef QB_SANITY
    if (c->samples_since_last < 1) {
      fprintf(QB_ERR, "B: qb_do_error_correction: cycle #%d: samples_since_last < 1 (%lld)\n",
              cix, c->samples_since_last);
      return QB_E_BUG;
    }
#endif
*/
  
    // pending run in (running_val, run_num_smps, run_start_smps, run_start_cycs)
    
   // value = (c->value == '0') ? 0 : 1;
    value = qb_atom_get_value_bool(c);

    if (value != running_val) {
    
      // end of run
      
      // so runs[0] is like [-2]
      // runs[1] has just finished so is like [-1]
      // and a brand new run is just beginning, stored in the variables, which is like [0]
      // we consider [-2] vs. [-1]
      
//~ printf("cyc %d: %u -> %u, run len %d\n",
       //~ cix, running_val, value, cix - run_start_cycs);
      
      // copy runs[1] to runs[0], then copy "live" variables to runs[1]
      runs[0] = runs[1]; // back up previous run
      
      runs[1].value          = running_val;
      runs[1].start_cycs     = run_start_cycs;
      runs[1].len_cycs       = cix - run_start_cycs;
      
//~ printf("runs[0].start,len,val=%d,%d,%u; runs[1].start,len,val=%d,%d,%u\n",
       //~ runs[0].start_cycs, runs[0].len_cycs, runs[0].value,
       //~ runs[1].start_cycs, runs[1].len_cycs, runs[1].value);
       
#ifdef QB_SANITY
      if ( (cix > 2) && ((runs[0].start_cycs + runs[0].len_cycs - 1) != (runs[1].start_cycs - 1) ) ) {
        fprintf(QB_ERR, "B: %s: wat (runs[0] final %d != runs[1] initial minus one %d)\n",
                QB_FUNC_M,
                (runs[0].start_cycs + runs[0].len_cycs - 1),
                (runs[1].start_cycs - 1));
        return QB_E_BUG;
      }
#endif
      
      //~ printf("at cyc #%d; end of run, smp %lld, len %lf cycs, rounded %d cycs\n",
             //~ cix, c->sample_num, runs[1].len_cycs_float, runs[1].len_cycs_int);

      // start new pending run in "live" variables
      running_val    = value;
      run_start_cycs = cix;
      
      // examine run lengths in cycles

      // whichever is the 0-run, we're only interested in correcting
      // if it's less than 22 cycles long (i.e. 11 bits), for 8P2 (=> 011111111SSP, where S, S, P all 1)
      
      zero_run_is_short =    ( (0 == runs[0].value) && (runs[0].len_cycs <= 22) )   // first run is 1200 Hz
                          || ( (0 == runs[1].value) && (runs[1].len_cycs <= 22) );  // second run is 1200 Hz
                          
      odd_adjacent_pair = (runs[0].len_cycs & 1) && (runs[1].len_cycs & 1);
      
      // final term (runs[1].start_cycs >= 1) is to prevent potential underflow
      if ( odd_adjacent_pair && zero_run_is_short && (runs[1].start_cycs >= 1) ) {
        
        // => adjacent pair of runs with odd numbers of cycles
      
      /*
        //~ printf("Odd pair smps(%lld, %lld), cycs(%d, %d), len_cycs(%.1lf, %.1lf), vals(%u,%u)\n",
        printf("Odd pair cycs(%d, %d), vals(%u, %u)\n",
               //~ runs[0].start_smps,     runs[1].start_smps,
               runs[0].start_cycs,     runs[1].start_cycs,
               //~ runs[0].len_cycs_float, runs[1].len_cycs_float,
               runs[0].value,          runs[1].value);
      */
               
        // OK. fix runs.
        // we either need to flip the final bit (cycle) of runs[0],
        // or the first bit (cycle) of runs[1]
        
        //~ printf("looking at cycles (%d, %d)\n", (runs[1].start_cycs - 1), runs[1].start_cycs);
        // final cycle of runs[0]
        c0 = cycs + ((size_t) runs[1].start_cycs - 1);
        //~ c0 = cycs +  (runs[0].start_cycs + runs[0].len_cycs - 1);
        // first cycle of runs[1]
        c1 = cycs + (size_t) runs[1].start_cycs;
        
        //early_value = ('0' == c0->value) ? 0 : 1;
        //late_value  = ('0' == c1->value) ? 0 : 1;
        
        early_value = qb_atom_get_value_bool(c0);
        late_value  = qb_atom_get_value_bool(c1);
        
//~ printf("smps=%lld,%lld, early,late=%u,%u\n", c0->sample_num, c1->sample_num, early_value, late_value);

        if ( ! ( (0==early_value) ^ (0==late_value) ) ) {
          // this bit has already been flipped by a previous operation,
          // so skip it
          skipped++;
          continue;
        }
               
        if (QB_ERRCORR_SINGLETON_WINS == strategy) {
          if (1 == runs[0].len_cycs) {
            //~ printf("    -> flip late (singleton)\n");
            flip_cycle_value(c1, flips, 1, verbose);
          } else if (1 == runs[1].len_cycs) {
            //~ printf("    -> flip early (singleton)\n");
            flip_cycle_value(c0, flips, 0, verbose);
          }
        } else if (QB_ERRCORR_CONFIDENCE_WINS == strategy) {
          if (c0->confidence > c1->confidence) {
            // first run is better, so flip first bit of second run
            //~ printf("    -> flip late (confidence)\n");
            flip_cycle_value(c1, flips, 1, verbose);
          } else {
            // second run is better, so flip last bit of first run
            //~ printf("    -> flip early (confidence)\n");
            flip_cycle_value(c0, flips, 0, verbose);
          }
        } else if (QB_ERRCORR_ZERO_WINS == strategy) {
          if ( '0' == c0->value ) {
            //~ printf("    -> flip late (zero wins)\n");
            flip_cycle_value(c1, flips, 1, verbose);
            //~ flip_cycle_value(c1);
          } else {
            //~ printf("    -> flip early (zero wins)\n");
            flip_cycle_value(c0, flips, 0, verbose);
            //~ flip_cycle_value(c0);
          }
        } else if (QB_ERRCORR_ONE_WINS == strategy) {
          if ( '0' == c0->value ) {
            //~ printf("    -> flip early (one wins)\n");
            flip_cycle_value(c0, flips, 0, verbose);
          } else {
            //~ printf("    -> flip late (one wins)\n");
            flip_cycle_value(c1, flips, 1, verbose);
          }
        } else if (QB_ERRCORR_EARLY_WINS == strategy) {
          //~ printf("    -> flip late (early wins)\n");
          flip_cycle_value(c1, flips, 1, verbose);
        } else if (QB_ERRCORR_LATE_WINS == strategy) {
          //~ printf("    -> flip early (late wins)\n");
          flip_cycle_value(c0, flips, 0, verbose);
        } else {
          fprintf(QB_ERR, "B: %s: bad strategy %u\n",
                  QB_FUNC_M, strategy);
          return QB_E_BUG;
        }
        
      } // endif (odd run)
      
    } // endif (new run)

  } // next cycle
  
  /*
  printf("  Flips: early %u, late %u, false->true %u, true->false %u, skipped %u.\n",
         flips[0][0]+flips[0][1],
         flips[1][0]+flips[1][1],
         flips[0][0]+flips[1][0],
         flips[0][1]+flips[1][1],
         skipped);
  */
  
  *flipped_early_out  = flips[0][0]+flips[0][1];
  *flipped_late_out   = flips[1][0]+flips[1][1];
  *flipped_0_to_1_out = flips[0][0]+flips[1][0];
  *flipped_1_to_0_out = flips[0][1]+flips[1][1];
  *skipped_out        = skipped;
  
  return e;
  
}

static void flip_cycle_value (qb_atom_t *c, u32_t flips[2][2], u8_t late, u8_t verbose) {

  (flips[late][('0'==c->value) ? 0 : 1])++;
  
  if ('0' == c->value) {
    c->value = '1';
    // also fiddle the power values
    // we're flipping a 0 to a 1
    // so make the 0-power smaller than the 1-power to reflect this
    c->power0 = c->power1 * 0.6f;
    // also fix the confidence
    c->confidence = c->power1 - c->power0;
  } else {
    c->value = '0';
    // also fiddle the power values
    c->power1 = c->power0 * 0.6f;
    c->confidence = c->power0 - c->power1;
  }
  
  if (verbose) {
    printf ("    [%lld]: flipped bit\n", c->sample_num);
  }

}
