/*
 *  Quadbike 2
 *  Copyright (C) 2023 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/


#include "qbio.h"
#include "tapespeed.h"
#include "inspect.h"
#include "span.h"
#include "util.h"

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

// new in alpha-2

// Contraption(Icon)(B)[TapeSideNoScrews].wav has a 0.06s silent gap (~2700 samples)
#define QB_LEADER_OR_SILENCE_MIN_LEN_S 0.05f

static qb_err_t append_span (qb_span_t **spans,
                             s32_t *alloc,
                             s32_t *fill,
                             qb_span_t *span);
                              

u8_t qb_is_squawk (qb_span_t *span,
                   qb_span_t *prev_span) { // or NULL
  if (NULL == prev_span)                      { return 0; }
  if (QB_SPAN_TYPE_SILENT != prev_span->type) { return 0; }
  if (span->len > 1100)                       { return 0; }  // FIXME: use proper number (rate/FREQ_2) * (>) 40
  if (QB_SPAN_TYPE_DATA != span->type)        { return 0; }
  return 1;
}


void qb_span_set_type (qb_span_t *span, u8_t new_type) {
  if (new_type == span->type) {
    if (QB_SPAN_TYPE_INVALID == new_type) {
      return;
    }
    //fprintf(QB_ERR, "W: qb_span_set_type call fails to change span->type (%s), ignored\n",
    //        qb_span_get_type_text(span));
    return;
  }
  // -- not true any more ...
  /*
  if (new_type == QB_SPAN_TYPE_SILENT) {
    span->reverses_output_polarity = 1;
  } else {
    span->reverses_output_polarity = 0;
  }
  */
  span->type = new_type;
}
         

// tedious function which handles all of the span identification
// and amalgamation etc. shenanigans
qb_err_t qb_mark_spans (float *goertz[2],
                        float max_powerX,
                        s64_t num_smps,
                        s32_t rate,
                        double silence_on_frac,    // for determining when signal -> silent
                        double signal_on_frac,     // for determining when silent -> signal
                        double summed_silent_frac, // for determining whether signal spans should be post-converted to silent
                        qb_span_t **spans_out,
                        s32_t *num_spans_out) {

  s64_t i;
  s32_t j;
  qb_span_t sp;
  double silence_on_thresh, signal_on_thresh;
  s64_t silence_or_leader_min_len;
  double summed_silence_on_thresh;
  
  s32_t alloc, fill;
  s32_t alloc2, fill2;
  
  s64_t last_non_silent_smp;
  s64_t last_zero_smp;
  
  qb_span_t *spans, *spans2;
  qb_span_t dataspan;

  qb_err_t e;
  
  double data_start_adjust_leader;
  double data_start_adjust_silent;
  double data_end_delay;
  s64_t data_end_delay_smps;
  
  memset(&sp, 0, sizeof(qb_span_t));
  sp.detected_input_phase_ix = -1;
  
  qb_span_set_type (&sp, QB_SPAN_TYPE_INVALID);
  
  silence_or_leader_min_len = (s64_t) round (QB_LEADER_OR_SILENCE_MIN_LEN_S * (double) rate);
  
  // two different silence thresholds for off and on, because we need some hysteresis,
  // to prevent squawk spans starting early, prepending a phantom start bit
  silence_on_thresh = max_powerX * silence_on_frac;
  signal_on_thresh  = max_powerX * signal_on_frac;
  
  alloc = 0;
  fill = 0;
  
  last_non_silent_smp = -1;
  last_zero_smp = -1;
  
  *spans_out = NULL;
  *num_spans_out = 0;
  
  spans = NULL;
  
  // we'll do this in two phases:
  // i) the first phase establishes the leader and silent sections.
  // ii) anything that is not accounted for in (i) becomes a data section.
  
  // so:
  
  for (i=0; i < num_smps; i++) {
  
    u8_t have_signal;
    u8_t value;
    u8_t end_of_run;
    
    float p0, p1;
    
    end_of_run = 0;
    
    p0 = goertz[0][i];
    p1 = goertz[1][i];
    
    // hysteresis, to prevent squawk spans starting early => phantom start bit
    if ((QB_SPAN_TYPE_SILENT == sp.type) || (QB_SPAN_TYPE_INVALID == sp.type)) {
      have_signal = (p0 > signal_on_thresh)  || (p1 > signal_on_thresh);
    } else {
      have_signal = (p0 > silence_on_thresh) || (p1 > silence_on_thresh);
    }
    value = (p1 > p0);
  
    if (QB_SPAN_TYPE_SILENT == sp.type) {
      if (have_signal) {
        end_of_run = 1;
      }
    } else if (QB_SPAN_TYPE_LEADER == sp.type) {
      if ( have_signal && ! value ) {
        end_of_run = 1;
      }
    } else { // SPAN_TYPE_INVALID, need to decide what kind of span it is going forward
      if ((i - last_non_silent_smp)  >= silence_or_leader_min_len) {
        qb_span_set_type (&sp, QB_SPAN_TYPE_SILENT);
      } else if ((i - last_zero_smp) >= silence_or_leader_min_len) {
        qb_span_set_type (&sp, QB_SPAN_TYPE_LEADER);
      }
    }

    // 2.0.2
    // hack to combat scruss bug:
    // https://www.stardot.org.uk/forums/viewtopic.php?p=446499#p446499
    if (i==(num_smps-1)) {
      end_of_run = 1;
    }
    
    if (end_of_run) {
    
//printf("end of run: %lld\n", i);

      // add new run
      
      if (QB_SPAN_TYPE_SILENT == sp.type) {
        sp.start = last_non_silent_smp + 1;
      } else {
        sp.start = last_zero_smp + 1;
      }

//if (0 == *num_spans_out) {
//  printf("qb_mark_spans: sp.start = %lld\n", sp.start);
//  exit(0);
//}
      
      sp.len = i - sp.start;
      
      e = append_span (&spans, &alloc, &fill, &sp);
      if (QB_E_OK != e) { return e; } // spans already freed
      
      memset(&sp, 0, sizeof(qb_span_t));
      sp.detected_input_phase_ix = -1;

      qb_span_set_type(&sp, QB_SPAN_TYPE_INVALID); // for next run
      
      // update these; they're invalid now the run is over
      last_non_silent_smp = i;
      last_zero_smp = i;

    } // endif (end of run)
    
    if (have_signal) {
      last_non_silent_smp = i;
    }
    
    if ( have_signal && ! value ) {
      last_zero_smp = i;
    }
    
    /*
    // HACK: if a run ends one smp before the end of the file, then
    // just quit early, rather than trying to squeeze in another
    // span
    if (i == (num_smps - 2)) {
      printf("    W: [%lld] Span ends just before end of file at [%lld]; ignore rem.\n",
             i, (num_smps - 1));
      break;
    }
    */
    
  } // next sample

//exit(0);
  
  // OK, now re-process the spans, and add in the missing data spans
  
  alloc2 = 0;
  fill2 = 0;
  spans2 = NULL;
  
  // special case: there is a gap before the first span
  // deal with this first
  
  // or if spans is NULL, which is the case at this point if we only have
  // one data span that comprises the entire file
  if ((NULL == spans) || ((fill>0) && (spans[0].start != 0))) {
  
    memset(&dataspan, 0, sizeof(qb_span_t));
  
    qb_span_set_type (&dataspan, QB_SPAN_TYPE_DATA);
    dataspan.start = 0;
    dataspan.len   = (spans == NULL) ? num_smps : spans[0].start;
    dataspan.atoms  = NULL;
    dataspan.num_atoms = 0;
    dataspan.detected_input_phase_ix = -1;
    
    e = append_span (&spans2, &alloc2, &fill2, &dataspan);
    if (QB_E_OK != e) {
      if (spans != NULL) {
        qb_free(spans);
      }
      return e; // spans2 already freed
    }
    
  }
  
  for (j=0; j < fill; j++) {
  
    qb_span_t *span, *next;
    s64_t min_dataspan_len;
    
    span = spans + j;
    
    // first, copy the span to the output list
    e = append_span (&spans2, &alloc2, &fill2, span);
    if (QB_E_OK != e) {
      qb_free(spans);
      return e; // spans2 already freed
    }
    
    if (j < (fill - 1)) {
    
      next = span + 1;
      
      if (next->start != (span->start + span->len)) { // gap in spans?
      
        memset(&dataspan, 0, sizeof(qb_span_t));
      
        // gap in the spans will be filled with a data span
        qb_span_set_type (&dataspan, QB_SPAN_TYPE_DATA);
        dataspan.start = span->start + span->len;
        dataspan.len   = next->start - dataspan.start;
        dataspan.atoms  = NULL;
        dataspan.num_atoms = 0;
        dataspan.detected_input_phase_ix = -1;
        
        // OK, but we're not going to recognise data spans
        // that are shorter than a single meaningful data frame
        // which is 10? bits?
        // hmm. "Duck!(Firebird).wav" has a weird squawky thing during
        // early leader that's about 300 samples long. We'll use
        // 7 bits minimum length to catch this.
        
        min_dataspan_len = (s64_t) round (((double) rate) * (7.0 / QB_FREQ_1));

        if (dataspan.len >= min_dataspan_len) {
          // data span is long enough, so add it
          e = append_span (&spans2, &alloc2, &fill2, &dataspan);
          if (QB_E_OK != e) {
            qb_free(spans);
            return e; // spans2 already freed
          }
        } else {
          // we won't add this data span, so we need to smush
          // its length into one of the adjacent spans instead
          
          // for now we'll just add it to the previous span,
          // since that's simpler
          spans2[fill2-1].len += dataspan.len;
          
        }
        
      }
      
    } // endif (next span exists)
    
  } // next input span

  qb_free(spans);
  
  // examine final span and see if we need end padding (yes, we always do)
  // => one more span
  
  if (fill2 > 0) {
  
    memset(&dataspan, 0, sizeof(qb_span_t));
    
    dataspan.start = spans2[fill2-1].start + spans2[fill2-1].len;
    dataspan.len   = num_smps - dataspan.start;
    dataspan.detected_input_phase_ix = -1;
    
    if (dataspan.len > 0) {
      qb_span_set_type (&dataspan, QB_SPAN_TYPE_SILENT);
      dataspan.atoms  = NULL;
      dataspan.num_atoms = 0;
      e = append_span (&spans2, &alloc2, &fill2, &dataspan);
      if (QB_E_OK != e) {
        qb_free(spans);
        return e; // spans2 already freed
      }
    }
  }

  // review what we've done -- many of the data spans should
  // actually be silent. we've based our analysis so far on
  // instantaneous power, but now we're interested in the integrated
  // power across the entire span.
//#define QB_SUMMED_SILENT_POWER_FRACTION (QB_SILENT_POWER_FRACTION)
  summed_silence_on_thresh = summed_silent_frac * max_powerX;

  for (j=0; j < fill2; j++) {
    qb_span_t *span;
    float summed_power0, summed_power1;
    float p0, p1;
    summed_power0 = 0.0f;
    summed_power1 = 0.0f;
    span = spans2 + j;
    for (i=span->start; i < (span->start + span->len); i++) {
      p0 = goertz[0][i];
      p1 = goertz[1][i];
      summed_power0 += p0;
      summed_power1 += p1;
    }
    summed_power0 /= (float) (span->start + span->len);
    summed_power1 /= (float) (span->start + span->len);
    // by dividing by the span len, we've normalised back to the power
    // for a single sample, which we can compare to QB_SILENT_POWER_FRACTION
    // as before
    
//printf("summed_silence_on_thresh = %f\n", summed_silence_on_thresh);
    
    if (    (summed_power0 < summed_silence_on_thresh)
         && (summed_power1 < summed_silence_on_thresh)
         // 2.0.1: now only for data spans, controversial:
         && (span->type == QB_SPAN_TYPE_DATA) ) { // only for data spans? or leader too??
         //&& (span->type != QB_SPAN_TYPE_SILENT) ) {
      // convert data span to silent span
      //printf("converting span %d (start %lld, type %s) to silent span (cycs = %p, num_cycs = %d)\n",
      //       j, span->start, qb_span_get_type_text(span), span->atoms, span->num_atoms);
      if (NULL != span->atoms) { // not sure if this is needed
        qb_free(span->atoms);
        span->atoms = NULL;
      }
      span->num_atoms = 0; // set to zero and it will get fixed up later when the silent spans are populated with a cycle
      //span->type = QB_SPAN_TYPE_SILENT;
      qb_span_set_type (span, QB_SPAN_TYPE_SILENT);
    }
    //printf("span %d (start %lld, len %lld): power0/smp = %lf, power1/smp = %lf\n",
    //       j, span->start, span->len, summed_power0, summed_power1);
  }
  
  // combine adjacent spans of the same type  
  e = qb_span_amalgamate_spans (spans2, &fill2);
  
  // now we'll go through and tweak the data spans a little;
  // - data spans need to be extended by ~18 half-bits (use 20) so that
  //   we make absolutely sure we capture everything after the
  //   start bit (the byte could be all ones and we don't want
  //   such bytes misinterpreted as leader tone ...)
  // consider:
  // 00 dd dd dd dd dd dd dd dd 11 LLLLLLL ...
  // worst case:
  // 00 11 11 11 11 11 11 11 11 11 LLLLLLL ...
  //    ^^^^^^^^^^^^^^^^^^^^^^^
  //   this is actually the byte
  //   -- delay end of span by ~18 half bits
  
  // - data spans also need to start slightly earlier,
  //   so that we don't clip off the start of the first
  //   start bit half-cycle -- maybe by 0.5 half-bits
  
  // leader into data ...
  // bring data start forwards by (seconds): 0.5 half-bits, plus a bit extra:
  data_start_adjust_leader = (0.5 / QB_FREQ_2) + (2.0 / 8000.0);
  // silence into data ...
  // *delay* data start by 0.35 bits
  // this is to alleviate the "squawk phantom start bit" problem
  data_start_adjust_silent = data_start_adjust_leader;
//data_start_adjust_silent = -0.4 / QB_FREQ_1;
  // delay end of all data spans by (seconds): 20 half-bits
  data_end_delay = (20.0 / QB_FREQ_2) ;
  
  //a *= rate; // convert to samples
  
  // convert to num samples
  data_start_adjust_leader *= rate;
  data_start_adjust_silent *= rate;
  data_end_delay           *= rate;
  
  // and round off
  data_end_delay_smps           = (s64_t) round(data_end_delay);
  
  for (j=1; j < (fill2 - 1); j++) {
  
    qb_span_t *span, *next, *prev;
    s64_t sa;
    
    span = spans2 + j;
    next = span + 1;
    prev = span - 1;
    
    if (QB_SPAN_TYPE_DATA == span->type) {
    
      // -- leader into data:
      // before: xxxxDDDDxxxx
      // after:  xxxDDDDDDxxx
      
      // -- silence into data:
      // before: xxxxDDDDxxxx
      // after:  xxxxxDDDDxxx
      
      if (QB_SPAN_TYPE_LEADER == prev->type) {
        sa = (s64_t) data_start_adjust_leader;
      } else {
        sa = (s64_t) data_start_adjust_silent;
      }
      
      // fix start
      prev->len   -= sa;
      span->start -= sa;
      span->len   += sa;
      
      // fix end
      span->len   += data_end_delay_smps;
      next->start += data_end_delay_smps;
      next->len   -= data_end_delay_smps;
      
    }
  }
  
  // HACK: prevent 1-sample piece of silence
  // being inserted at very end of file (messes up CSW save)
  if ((QB_SPAN_TYPE_SILENT == spans2[fill2-1].type) && (spans2[fill2-1].len == 1)) {
    // 2.0.2: downgraded from W to M
    printf("    M: [%lld] Final span is one sample of silence; deleting.\n", spans2[fill2-1].start);
    fill2--;
  }
  
  
#ifdef QB_SANITY
  // there should now be no gaps -- each span should end
  // where the next one starts, so assert that
  for (j=0; j < (fill2 - 1); j++) {
    qb_span_t *span, *next;
    span = spans2 + j;
//printf("span %u type %s\n", j, qb_span_get_type_text(span));
    next = span + 1;
    if ((span->start + span->len) != next->start) {
      fprintf(QB_ERR, "B: %s: discontinuity between adjacent spans (%d and %d):\n"
                      "B: %s: span %d ends at %lld, span %d starts at %lld\n",
                      QB_FUNC_M,
                      j,
                      j+1,
                      QB_FUNC_M,
                      j,
                      span->start + span->len,
                      j+1,
                      next->start);
      qb_free(spans2);
      return QB_E_BUG;
    }
  }
#endif
  
  *spans_out = spans2;
  *num_spans_out = fill2;
  
  return QB_E_OK;
  
}


u8_t qb_span_has_viable_phase (qb_span_t *span) {
  if (NULL == span) { return 0; }
  if (QB_SPAN_TYPE_DATA != span->type) { return 0; }
  if (span->len < 1000) { return 0; }
  return 1;
}

qb_err_t qb_span_amalgamate_spans (qb_span_t *spans, s32_t *num_spans_inout) {

  // combine adjacent spans of same type
  // FIXME: not efficient (memmove)

  s32_t j;
  
  for (j=0; j < (*num_spans_inout - 1); j++) {
  
    qb_span_t *span, *next;
    qb_atom_t *p;
    
    span  = spans + j;
    next  = span + 1;
    
    if (span->type == next->type) {
    
      // coagulate spans
      span->len += next->len;
      
      if (span->detected_input_phase_ix != next->detected_input_phase_ix) {
        fprintf(QB_ERR, "W: coagulating spans (%s, %s) with different phases (%d, %d)\n",
                qb_span_get_type_text(span),
                qb_span_get_type_text(next),
                span->detected_input_phase_ix,
                next->detected_input_phase_ix);
      }
      
      // also have to fix up any cycles
      // this is different for silent and non-silent spans:
      
      if (QB_SPAN_TYPE_SILENT == span->type) {
      
        // silent spans are only allowed to contain a single cycle,
        // so we just delete the second silent span, but in this case
        // we need to fix up samples_since_last on the span following
        // the second silent span (span+2):
        
        // so
        
        // | silent span 1 | silent span 2 | non-silent span 3
        // |    "span"     |     "next"    |     "span + 2"
        // |               |               |
        // | C             | C             | C C C C C ......
        //       delete this span ^^^        ^ fix samples_since_last on this cycle
        
        
        //if ((j < (*num_spans_inout - 2)) && (0 != (span+2)->num_atoms) && (0 != span->num_atoms)) {
        //  (span+2)->atoms[0].samples_since_last = (span+2)->atoms[0].sample_num - span->atoms[0].sample_num;
        //}
      
      } else {
      
        // for non-silent span types we need to move the second
        // span's cycles into the first span
      
        if (    (next->num_atoms > 0)
             && (NULL != next->atoms)
             && (NULL != span->atoms)) {
          p = qb_realloc(span->atoms, sizeof(qb_atom_t) * (next->num_atoms + span->num_atoms));
          if (NULL == p) {
            fprintf(QB_ERR, "E: Out of memory amalgamating spans.\n");
            return QB_E_MALLOC;
          }
          span->atoms = p;
          memcpy(span->atoms + span->num_atoms,
                 next->atoms,
                 sizeof(qb_atom_t) * next->num_atoms);
          span->num_atoms += next->num_atoms;
        }
        
      }
      
          
//printf("combining spans #%d (start %lld) and #%d (start %lld)\n",
//       j, spans2[j].start, j+1, spans2[j+1].start);
      memmove (spans + j + 1,
               spans + j + 2,
               sizeof(qb_span_t) * (*num_spans_inout - (j+2)));
      (*num_spans_inout)--;
      // need to re-process this span, in case the next span
      // is also of the same type (otherwise it will be missed ...)
      j--;
    }
  }
  
  return QB_E_OK;
}

#define QB_MAX_ATOMS_PER_SPAN 100000000

static qb_err_t append_span (qb_span_t **spans,
                              s32_t *alloc,
                              s32_t *fill,
                              qb_span_t *span) {
  qb_span_t *p;
  
//printf("append_span, len %lld smps\n", span->len);
  
#ifdef QB_SANITY
  if (span->num_atoms > QB_MAX_ATOMS_PER_SPAN) {
    fprintf(QB_ERR, "B: %s: span has insane num_cycs %d\n", QB_FUNC_M, span->num_atoms);
    return QB_E_BUG;
  }
#endif
  // realloc if needed
  if (*fill >= *alloc) {
    *alloc = *fill + 50;
    p = qb_realloc (*spans, sizeof(qb_span_t) * *alloc);
    if (NULL == p) {
      fprintf(QB_ERR, "E: out of memory allocating spans.\n");
      if (NULL != *spans) {
        qb_free(*spans);
        *spans = NULL;
      }
      return QB_E_MALLOC;
    }
    // fill new section with sentinel garbage
    //memset(p + *fill, 0x0, (*alloc - *fill) * sizeof(qb_span_t));
    *spans = p;
  }
  (*spans)[*fill] = *span;
  (*fill)++;
  return QB_E_OK;
}

const char *qb_span_get_type_text (qb_span_t *s) {
  if (QB_SPAN_TYPE_DATA == s->type) {
    return "data";
  } else if (QB_SPAN_TYPE_LEADER == s->type) {
    return "leader";
  } else if (QB_SPAN_TYPE_SILENT == s->type) {
    return "silent";
  }
  return "???";
}



qb_err_t qb_spans_populate_tape_speeds (qb_span_t *spans,
                                        s32_t num_spans,
                                        s32_t rate,
                                        float *src_f,
                                        float *min_speed_out,
                                        float *max_speed_out,
                                        u8_t display_progress,
                                        u8_t verbose,
                                        qb_inspect_t *inspect) {
                                        
  qb_err_t e;
  s32_t sn;
  s32_t min_hi_f_2400, max_hi_f_2400;
  s32_t min_hi_f_1200, max_hi_f_1200;
  s32_t min_speed_span, max_speed_span;
  float last_valid_tapespeed;
  
//#ifdef QB_SANITY
//  if (num_spans < 2) {
//    fprintf(QB_ERR, "B: %s: num_spans < 2\n", QB_FUNC_M);
//    return QB_E_BUG;
//  }
//#endif

  e = QB_E_OK;
  
  *min_speed_out = 10000.0;
  *max_speed_out = 0.0;
  
  min_speed_span = 0;
  max_speed_span = 0;
  
  // we measure over a range that is wider than what we subsequently allow
  min_hi_f_2400  = (s32_t) round (QB_TAPE_SPEED_SEARCH_MIN * QB_FREQ_2);
  max_hi_f_2400  = (s32_t) round (QB_TAPE_SPEED_SEARCH_MAX * QB_FREQ_2);
  // if needed, we may need to scan at 1200 Hz too:
  min_hi_f_1200  = (s32_t) round (QB_TAPE_SPEED_SEARCH_MIN * QB_FREQ_1);
  max_hi_f_1200  = (s32_t) round (QB_TAPE_SPEED_SEARCH_MAX * QB_FREQ_1);
  
  // we do this manually here in order to avoid its
  // printed output from messing up the progress display
  if (inspect->enabled) {
    e = qb_inspect_open_if_needed (inspect->files + QB_INSPECT_FILE_IX_SPAN_TAPESPEEDS);
    if (QB_E_OK != e) { return e; }
  }

  printf("    Tape speed: ");
  fflush(stdout);
  if (display_progress) { printf("                "); }   //xx% 1.xxx
  
  last_valid_tapespeed = 1.0;
  
//printf("num_spans = %d\n", num_spans);
  
  for (sn=0; sn < num_spans; sn++) {
  
    s64_t z;
    float speed_1200, speed_2400;
    char *speed_src;
    
    speed_src = "      ";
    
#ifdef QB_SANITY
    if (spans[sn].len < 0) {
      fprintf(QB_ERR, "\nB: %s: span %u len is negative (%lld)\n", QB_FUNC_M, sn, spans[sn].len);
      return QB_E_BUG;
    }
#endif
    
    // this proved unreliable, so now we'll test (data) spans
    // for both 2400 and 1200 Hz power, and use the value that
    // is closest to nominal (1.0)
    
    speed_1200 = 0.0f;
    
    if ( QB_SPAN_TYPE_SILENT != spans[sn].type ) {
      // only data spans have 1200 Hz in them, so only
      // evaluate 1200 power for data spans
      if ( QB_SPAN_TYPE_DATA == spans[sn].type ) {
        e = qb_span_measure_speed  (rate,
                                    src_f + spans[sn].start,
                                    spans[sn].len,
                                    (float) min_hi_f_1200,
                                    (float) max_hi_f_1200,
                                    QB_FREQ_1,
                                    &speed_1200);
        if (QB_E_OK != e) { break; }
      }
      e = qb_span_measure_speed  (rate,
                                  src_f + spans[sn].start,
                                  spans[sn].len,
                                  (float) min_hi_f_2400,
                                  (float) max_hi_f_2400,
                                  QB_FREQ_2,
                                  &speed_2400);
      if (QB_E_OK != e) { break; }
      // pick the value (1200 vs 2400) that is less deviant
      if (fabsf(1.0f - speed_1200) > fabsf(1.0f - speed_2400)) {
        spans[sn].speed = speed_2400;
        speed_src = "(2400)";
      } else {
        spans[sn].speed = speed_1200;
        speed_src = "(1200)";
      }
      if ( ! qb_is_tape_speed_legal (spans[sn].speed) ) {
        // bad speed
        if (verbose) {
          //fprintf (QB_ERR, "\n    [%lld] W: bad tape speed %.3lf on span %d; using %.3lf\n",
          fprintf (QB_ERR, "\n    [%lld] W: bad tape speed %.3f on span %d; using %.3f\n",
                   spans[sn].start, spans[sn].speed, sn, last_valid_tapespeed);
          printf("    Tape speed: ");
          fflush(stdout);
          if (display_progress) { printf("                "); }   //xx% 1.xxx
        }
        spans[sn].speed = last_valid_tapespeed;
      } else {
        last_valid_tapespeed = spans[sn].speed;
      }
      if (spans[sn].speed > *max_speed_out) {
        *max_speed_out = spans[sn].speed;
        max_speed_span = sn;
      }
      if (spans[sn].speed < *min_speed_out) {
        *min_speed_out = spans[sn].speed;
        min_speed_span = sn;
      }
    } else { // span is silent
      //spans[sn].speed = 1.0;
      spans[sn].speed = 1.0f;
    }
    
    if (inspect->enabled) {
      s16_t *speedbuf;
      float f;
      float range;
      s16_t f16;
      range = (float) (QB_TAPE_SPEED_SEARCH_MAX - QB_TAPE_SPEED_SEARCH_MIN);
      speedbuf = qb_malloc (sizeof(s16_t) * spans[sn].len);
      f = (spans[sn].speed - 1.0f) / range;
      f16 = qb_float_to_s16(f);
      for (z=0; z < spans[sn].len; z++) {
        speedbuf[z] = f16;
      }
      e = qb_inspect_append_s16_buf (inspect->files + QB_INSPECT_FILE_IX_SPAN_TAPESPEEDS,
                                     speedbuf,
                                     spans[sn].len);
      qb_free(speedbuf);
      speedbuf = NULL;
      if (QB_E_OK != e) { break; }
    }

    //printf ("sp#%d: %s; smps %llu -> %llu; len %llu smps",
    //        spans[sn].ix, s, spans[sn].start, spans[sn].start + spans[sn].len, spans[sn].len);
    //    printf("\b\b\b");
    //printf("%2lld%%", starting_percent + (s64_t) floor((pos * 100.0f * fraction_of_total_job) / (float) len));
    if (display_progress) {
      printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b");
    
      if ( QB_SPAN_TYPE_SILENT != spans[sn].type ) {
        printf("%2lld%% %.3f %s",
               (s64_t) floor((sn * 100.0f) / (float) num_spans),
               spans[sn].speed,
               speed_src);
      } else {
        printf("%2lld%%             ",
               (s64_t) floor((sn * 100.0f) / (float) num_spans));
      }
      
      fflush(stdout); // MacOS
      
    }
    
//printf("span tapespeed %lf\n", spans[sn].speed);
    
    //if ( QB_SPAN_TYPE_SILENT != spans[sn].type ) {
    //  printf(", speed %.3f", spans[sn].speed);
   // }
    //putchar('\n');
  } // next span
  
  if (QB_E_OK != e) {
    printf("\n");
    return e;
  }
  
  if (display_progress) {
    printf("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b");
    fflush(stdout);
  }
  printf("min %.3f (span #%d/%d), max %.3f (span #%d/%d); done.\n",
         *min_speed_out,
         min_speed_span, num_spans,
         *max_speed_out,
         max_speed_span, num_spans);
  fflush(stdout);
  
  return QB_E_OK;
}


void qb_span_finish (qb_span_t *span) {
  if (NULL != span->atoms) {
    qb_free(span->atoms);
    span->atoms = NULL;
    span->num_atoms = -1;
  }
}

s64_t qb_get_span_start_accurate_or_rough (qb_span_t *span) {
  // return either the first cycle's time, or
  // if we don't have any cycles yet, the rough start-of-span number
  if (NULL == span->atoms) {
    return span->start;
  }
  return span->atoms[0].sample_num;
}

void qb_span_deal_w_float_smpnum_field (qb_span_t *span, u8_t integer_to_float) {
  s32_t an;
  for (an=0; an < span->num_atoms; an++) {
    // copy integer sample nums to the floating point field
    if (integer_to_float) {
      // integer -> float
      span->atoms[an].tmp_smpnum_f = (double) span->atoms[an].sample_num;
    } else {
      // float -> integer
      span->atoms[an].sample_num = (s64_t) round(span->atoms[an].tmp_smpnum_f);
      span->atoms[an].tmp_smpnum_f = NAN;
    }
  }
}


qb_err_t qb_spans_count_and_maybe_fix_odd_runs_by_atom_insert (qb_span_t *spans,
                                                              s32_t num_spans,
                                                              s64_t total_input_len,
                                                              s32_t rate,
                                                              u8_t fix_them,
                                                              s64_t *num_data_zeroval_odd_runs_out,
                                                              s64_t *num_data_oneval_odd_runs_out,
                                                              s64_t *num_squawk_zeroval_odd_runs_out,
                                                              s64_t *num_squawk_oneval_odd_runs_out,
                                                              u8_t verbose) {

  s32_t sn;
  qb_err_t e;
  s32_t num_inserted_atoms;
  
  e = QB_E_OK;
  
  *num_data_zeroval_odd_runs_out   = 0;
  *num_data_oneval_odd_runs_out    = 0;
  *num_squawk_zeroval_odd_runs_out = 0;
  *num_squawk_oneval_odd_runs_out  = 0;
  
  num_inserted_atoms = 0;
  
  // FIXME: should be zero-valued odd runs only (for now)??
  
  for (sn=0; sn < num_spans; sn++) {
  
    qb_span_t *span;
    s32_t an;
    s32_t run_len;
    u8_t run_value;
    s32_t run_start;
    u8_t leader;
    u8_t final_span;
    u8_t is_squawk;
    
    span = spans + sn;
    
    // just skip non-data spans
    if (    (QB_SPAN_TYPE_DATA != span->type) ) {
         //|| ) { // do not correct squawks
         //|| ! span->transient_pll_lock_dropout) {
      continue;
    }
    
    final_span = (sn >= (num_spans - 1));
    is_squawk  = qb_is_squawk(span, (sn>0)?(span-1):NULL);
    
    //printf("[%lld] data span %d: transient PLL lock dropout, scanning for odd runs ...\n",
    //       qb_get_span_start_accurate_or_rough(span), sn);
    
    run_len   = 0;
    run_value = 1;
    run_start = 0;
    leader    = 1;
    
    if (fix_them && ! is_squawk) {
      // we use the floating point copy of the smpnum for the duration
      // of this function, so populate it:
      qb_span_deal_w_float_smpnum_field (span, 1); // 1 = int -> float
    }

    // CAREFUL: span->num_atoms may increase as this loop runs!
    for (an=0; an < span->num_atoms; an++) {
    
      qb_atom_t *cyc;
      u8_t value; // value doesn't exist on cycles yet
      
      // CAREFUL: qb_span_insert_atom() may realloc span->atoms while this loop runs!
      cyc = span->atoms + an;
      
      value = qb_atom_get_value_bool (cyc);
      
      // flush any remaining 2400-cycles from start of span;
      // span must start with a 0 start bit
      if (leader) {
        if (value == 0) {
          leader = 0;
        } else {
          continue;
        }
      }
      
      if (value != run_value) {
      
        // end of run
        
        // odd run?
        if (run_len & 1) { // && (run_value == 0)) {
        
          //printf("M: [%lld] odd run (value %u, len %d cycs)\n",
          //       (s64_t) round(span->atoms[run_start].tmp_smpnum_f), run_value, run_len);
          
          //if ( ! is_squawk ) {
          //  printf("M: [%lld] data span odd run (value %u, len %d atoms)\n",
          //         span->atoms[run_start].sample_num, run_value, run_len);
          //}
          
          // increment appropriate counter
          if (is_squawk) {
            if (run_value) {
              (*num_squawk_oneval_odd_runs_out)++;
            } else {
              (*num_squawk_zeroval_odd_runs_out)++;
            }
          } else {
            if (run_value) {
              (*num_data_oneval_odd_runs_out)++;
            } else {
              (*num_data_zeroval_odd_runs_out)++;
            }
          }
          
          if (fix_them && ! is_squawk) {
            
            e = qb_span_insert_atom (span,
                                     final_span ? NULL : (span+1),
                                     total_input_len,
                                     run_start,
                                     // length of atom to insert (use nominal):
                                     (s64_t) (round((double) rate) / (QB_FREQ_2 * span->speed)),
                                     // this '1' value written here has no meaning;
                                     // remember that atom->value is mostly unused now;
                                     // values on atoms are read by calling qb_atom_get_value_bool(atom),
                                     // which just compares powers, so this has no significance
                                     // and should be ignored:
                                     '1',
                                     verbose);
            if (QB_E_OK != e) { break; }
            
            // we inserted an atom, so we need to +1
            an++;
            num_inserted_atoms++;
            
          } // endif (fix them)

        } // endif (odd run)
        
        // set up next run
        run_len   = 1;
        run_value = value;
        run_start = an;
        
      } else { // not end of run
        run_len++;
      }

    } // next cycle
    
    if (fix_them && ! is_squawk) {
      // copy float values back to integer atom.sample_num field
      // float field now becomes invalid
      qb_span_deal_w_float_smpnum_field (span, 0);
    }
    
    if (QB_E_OK != e) { break; }
    
  } // next span
  
  if (fix_them) {
    printf("    Odd runs (correction): Inserted %d atoms.\n", num_inserted_atoms);
  }
  
  return e;
  
}



//#define QB_SPAN_INSERT_ATOM_PRINT_DEBUG

qb_err_t qb_span_insert_atom (qb_span_t *span,
                               qb_span_t *next,
                               s64_t total_input_len,
                               s32_t run_start_atom_num,
                               s64_t new_cyc_len,
                               u8_t value,
                               u8_t verbose) {

  s64_t atom_len_nom;
  double atom_len_nom_f;
  s64_t insert_point_smps;
  s64_t pre_smps, post_smps, pre_smps_extra, post_smps_extra;
  double pre_frac, post_frac;
  s64_t start_smps, end_smps;
  double len_f;
  s32_t pre_atoms, post_atoms;
  double pre_extra_per_atom, post_extra_per_atom;
  double pre_accumulator, post_accumulator;
  s32_t an;
  qb_atom_t *p;
  qb_atom_t atom;
  
#ifdef QB_SPAN_INSERT_ATOM_PRINT_DEBUG
  if (QB_SPAN_TYPE_SILENT == span->type) {
    printf("before:\n");
    for (an=0; an < span->num_atoms; an++) {
      printf("  atom %d @ [%lf], value %c\n",
             an, span->atoms[an].tmp_smpnum_f, span->atoms[an].value);
    }
  }
#endif
  
  // one nominal atom len -- we'll split this between pre- and post- sections
  atom_len_nom = new_cyc_len;
  atom_len_nom_f = (double) atom_len_nom;
  
//printf("nominal = %lld smps\n", atom_len_nom);
  
  // find the point to insert the new atom
  insert_point_smps = (s64_t) round(span->atoms[run_start_atom_num].tmp_smpnum_f);
  
  // get first and last atom times in this span
  start_smps = (s64_t) round(span->atoms[0].tmp_smpnum_f);
  if (next != NULL) {
    end_smps = qb_get_span_start_accurate_or_rough(next);
  } else {
    end_smps = total_input_len;
  }

  // work out default lengths of pre- and post- sections, in samples
  pre_smps  = insert_point_smps - start_smps;
  post_smps = end_smps - insert_point_smps;

  // first atom to last atom: floating-point length
  len_f = (double) (end_smps - start_smps);
  
  // get pre- and post- lengths as fractions of the overall span length
  pre_frac  = ((double) pre_smps)  / len_f;
  post_frac = ((double) post_smps) / len_f;

/*
s64_t a,b,c,d,e,f,g;
a = span->atoms[run_start_atom_num-3].sample_num;
b = span->atoms[run_start_atom_num-2].sample_num;
c = span->atoms[run_start_atom_num-1].sample_num;
d = span->atoms[run_start_atom_num].sample_num;
e = span->atoms[run_start_atom_num+1].sample_num;
f = span->atoms[run_start_atom_num+2].sample_num;
g = span->atoms[run_start_atom_num+3].sample_num;
printf("before: [ %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld ]\n",
       a, (b-a), b, (c-b), c, (d-c), d, (e-d), e, (f-e), f, (g-f), g); */
  
//printf("pre_frac = %lf, post_frac = %lf\n", pre_frac, post_frac);

  // split the new atom's samples between pre and post sections
  pre_smps_extra  = (s64_t) round (atom_len_nom_f * pre_frac );
  post_smps_extra = (s64_t) round (atom_len_nom_f * post_frac);

  // now work out how much extra to add to each atom
  // get pre- and post- section lengths in atoms
  pre_atoms  = run_start_atom_num;
  post_atoms = span->num_atoms - run_start_atom_num; // this is +1 because we have the length of the final atom, too

  //           |-> next span
  // A.........|A
  //            ^ end_smps
  // ^ run_start_atom_num = 0
  // ^ start_smps
  // pre_atoms = 0, post_atoms = 1
  
  pre_extra_per_atom  = ((double) pre_smps_extra)  / (double) pre_atoms;
  post_extra_per_atom = ((double) post_smps_extra) / (double) post_atoms;
  
  pre_accumulator  = 0.0;
  post_accumulator = 0.0;

#ifdef QB_SPAN_INSERT_ATOM_PRINT_DEBUG
  printf("end_smps = %lld\n", end_smps);
  printf("pre_smps = %lld, post_smps = %lld\n", pre_smps, post_smps);
  printf("len_f = %lf\n", len_f);
  printf("pre_frac = %lf, post_frac = %lf\n", pre_frac, post_frac);
  printf("pre: +%lld samples; post: +%lld samples\n",
         (pre_smps_extra),
         (post_smps_extra));
  printf("pre_atoms = %d, post_atoms = %d\n", pre_atoms, post_atoms);
  printf("pre_extra_per_atom = %lf, post_extra_per_atom = %lf\n",
         pre_extra_per_atom, post_extra_per_atom);
#endif
  
  // pre- section: work forwards, moving atoms earlier
  for (an = 0; an < run_start_atom_num; an++) {
    double d;
    pre_accumulator += pre_extra_per_atom;
    d = span->atoms[an].tmp_smpnum_f - pre_accumulator;
    if (round(d) < span->start) {
      if (verbose) {
        fprintf(QB_ERR, "  W: insertion would move atom off span start [%lld < %lld]; ignoring\n",
                (s64_t) round(d), span->start);
      }
    } else {
      span->atoms[an].tmp_smpnum_f = d;
    }
  }
  
  // post- section: work backwards, moving atoms later
  for (an = (span->num_atoms - 1); an >= run_start_atom_num; an--) {
    double d;
    // moved this from after the smpnum correction
    post_accumulator += post_extra_per_atom;
    d = span->atoms[an].tmp_smpnum_f + post_accumulator;
    if (round(d) >= (span->start + span->len)) {
      if (verbose) {
        fprintf(QB_ERR, "  W: insertion would move atom off span end [%lld >= %lld]; ignoring\n",
                (s64_t) round(d), span->start + span->len);
      }
      //return QB_E_BUG;
    } else {
      span->atoms[an].tmp_smpnum_f = d;
    }

  }
  
//printf("final: new gap is %lld smps\n", span->atoms[run_start_atom_num].sample_num - span->atoms[run_start_atom_num-1].sample_num);
  
//printf("final: pre_accumulator = %lf, post_accumulator = %lf\n",
  //     pre_accumulator, post_accumulator);
  
  // make space
  // AAAAA, len 5
  //   ^ run_start_atom_num
  
  p = qb_realloc (span->atoms, sizeof(qb_atom_t) * (span->num_atoms + 1));
  if (NULL == p) {
    fprintf(QB_ERR, "  E: Out of memory extending span atoms buffer.\n");
    return QB_E_MALLOC;
  }
  span->atoms = p;
  
  memmove (span->atoms + run_start_atom_num + 1,
           span->atoms + run_start_atom_num,
           sizeof(qb_atom_t) * (span->num_atoms - run_start_atom_num));
  
  // finally we need to insert one extra atom here, at the original
  // run_start_atom_num position; base it on a copy of the original
  // atom that was at this point
  memcpy (&atom,
          span->atoms + run_start_atom_num + 1,
          sizeof(qb_atom_t));
          
  (span->num_atoms)++;
  
  atom.value = value;
  
  // assign sample number to the new atom
  if ( (0 == run_start_atom_num) || (run_start_atom_num >= (span->num_atoms - 1)) ) {
    // avoid buffer over/underrun
    // just stick the new atom where the old one was
    atom.tmp_smpnum_f = (double) insert_point_smps;
  } else {
    // otherwise interpolate between atoms either side
    atom.tmp_smpnum_f = (span->atoms[run_start_atom_num+1].tmp_smpnum_f + span->atoms[run_start_atom_num-1].tmp_smpnum_f) / 2.0;
  }
  atom.inserted = 1;

#ifdef QB_SPAN_INSERT_ATOM_PRINT_DEBUG
printf("inserted new %c-atom at [%lld]\n", value, (s64_t) round(atom.tmp_smpnum_f));
#endif

/*
printf("before insert (atom %d): %lld <%lld> %lld <%lld> %lld\n",
       run_start_atom_num,
       span->atoms[run_start_atom_num-1].sample_num,
       span->atoms[run_start_atom_num].sample_num - span->atoms[run_start_atom_num-1].sample_num,
       span->atoms[run_start_atom_num].sample_num,
       span->atoms[run_start_atom_num+1].sample_num - span->atoms[run_start_atom_num].sample_num,
       span->atoms[run_start_atom_num+1].sample_num); */

  span->atoms[run_start_atom_num] = atom;
  
/*
printf("after insert (atom %d): %lld <%lld> %lld <%lld> %lld\n",
       run_start_atom_num,
       span->atoms[run_start_atom_num-1].sample_num,
       span->atoms[run_start_atom_num].sample_num - span->atoms[run_start_atom_num-1].sample_num,
       span->atoms[run_start_atom_num].sample_num,
       span->atoms[run_start_atom_num+1].sample_num - span->atoms[run_start_atom_num].sample_num,
       span->atoms[run_start_atom_num+1].sample_num); */

/*
  s64_t x,y,z;
  x = span->atoms[run_start_atom_num-1].sample_num;
  y = span->atoms[run_start_atom_num].sample_num;
  z = span->atoms[run_start_atom_num+1].sample_num;
  printf("interpolated: (%lld) <%lld> %lld <%lld> (%lld)\n",
         x,
         y-x,
         y,
         z-y,
         z); */
  
  
/*
a = span->atoms[run_start_atom_num-3].sample_num;
b = span->atoms[run_start_atom_num-2].sample_num;
c = span->atoms[run_start_atom_num-1].sample_num;
d = span->atoms[run_start_atom_num].sample_num;
e = span->atoms[run_start_atom_num+1].sample_num;
f = span->atoms[run_start_atom_num+2].sample_num;
g = span->atoms[run_start_atom_num+3].sample_num;
printf("after: [ %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld ]\n",
       a, (b-a), b, (c-b), c, (d-c), d, (e-d), e, (f-e), f, (g-f), g); */
  
/*
s64_t a,b,c,d,e,f,g;
a = span->atoms[span->num_atoms-7].sample_num;
b = span->atoms[span->num_atoms-6].sample_num;
c = span->atoms[span->num_atoms-5].sample_num;
d = span->atoms[span->num_atoms-4].sample_num;
e = span->atoms[span->num_atoms-3].sample_num;
f = span->atoms[span->num_atoms-2].sample_num;
g = span->atoms[span->num_atoms-1].sample_num;
printf("end is now: [ %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld <%lld> %lld ]\n",
       a, (b-a), b, (c-b), c, (d-c), d, (e-d), e, (f-e), f, (g-f), g); */
  
//exit(0);

// MEH
// display last 20 atoms
/*
printf("from atom %d: ", (span->num_atoms - 21));
for (an=(span->num_atoms - 21); an < (span->num_atoms - 1); an++) {
  printf("%lld ", span->atoms[an+1].sample_num - span->atoms[an].sample_num);
}
printf("\n");

// sanity
for (an=0; an < (span->num_atoms - 1); an++) {
  if (span->atoms[an+1].sample_num <= span->atoms[an].sample_num) {
    fprintf(QB_ERR, "B: fail: atom %d/%d: sample num non-monotonic: (%lld) (%lld) %lld, %lld (%lld) (%lld)\n",
            an, (span->num_atoms-1),
            span->atoms[an-2].sample_num,
            span->atoms[an-1].sample_num,
            span->atoms[an].sample_num,
            span->atoms[an+1].sample_num,
            span->atoms[an+2].sample_num,
            span->atoms[an+3].sample_num);
    return QB_E_BUG;
  }
}
*/

#ifdef QB_SPAN_INSERT_ATOM_PRINT_DEBUG
  if (QB_SPAN_TYPE_SILENT == span->type) {
    printf("after:\n");
    for (an=0; an < span->num_atoms; an++) {
      printf("  atom %d @ [%lf], value %c\n", an, span->atoms[an].tmp_smpnum_f, span->atoms[an].value);
    }
  }
  printf("\n\n");
#endif
  
  return QB_E_OK;

}
