/*
 *  Quadbike 2
 *  Copyright (C) 2023 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/
#include "sync.h"

#include "sync_pll.h"
#include "sync_goertzel.h"
#include "sync_walk.h"

qb_err_t qb_get_sync_for_data_spans  (u8_t sync_method,
                                      s64_t srclen_smps,
                                      s32_t rate,
                                      qb_span_t *spans,
                                      s32_t num_spans,
                                      qb_pll_carrier_t pll_carriers[4], // for PLL only
                                      qb_inspect_t *inspect,
                                      u8_t display_progress,
                                      float *goertz[2],
                                      float max_powerX,
                                      qb_wave_feature_t *feats,         // for walk mode only
                                      s64_t num_feats,                  // for walk mode only
                                      u8_t verbose,
                                      u8_t freq_allow_odd_runs) {      // freq mode only -- even run len constraint

  qb_err_t e;
  s32_t sn;
  
  e = QB_E_OK;
  
  if (QB_SYNC_METHOD_PLL == sync_method) {
  
    e = qb_derive_sync_pll (srclen_smps,
                            rate,
                            spans,
                            num_spans,
                            pll_carriers,
                            inspect,
                            display_progress,
                            verbose);
    
    // verify lock qualities

    for (sn=0; (QB_E_OK == e) && (sn < num_spans); sn++) {
      qb_span_t *span;
      span = spans + sn;
      //if ( verbose && (QB_SPAN_TYPE_SILENT != span->type) && (span->pll_lock_quality < 0.3) ) {
      if ( verbose && (QB_SPAN_TYPE_SILENT != span->type) && (span->pll_lock_quality < 0.3f) ) {
        //fprintf(QB_ERR, "    W: [%lld] %s span #%d: poor PLL lock quality: %lf\n",
        fprintf(QB_ERR, "    W: [%lld] %s span #%d: poor PLL lock quality: %f\n",
                qb_get_span_start_accurate_or_rough(span),
                qb_span_get_type_text(span),
                sn,
                span->pll_lock_quality);
      }
    }
 
  } else if (QB_SYNC_METHOD_WALK == sync_method) {

    e = qb_sync_walk (feats,
                      num_feats,
                      rate,
                      spans,
                      num_spans);
    
  } else if (QB_SYNC_METHOD_GOERTZEL == sync_method) {
    e = qb_sync_freq (rate,
                      spans,
                      num_spans,
                      //goertzel,
                      goertz,
                      max_powerX,
                      inspect,
                      ! freq_allow_odd_runs,
                      verbose);
  } else {
    fprintf(QB_ERR, "B: %s: Unknown sync method %u.\n", QB_FUNC_M, sync_method);
    return QB_E_BUG;
  }
    
  return e;
  
}
