/*
 *  Quadbike 2
 *  Copyright (C) 2023 'Diminished'

 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include "audio.h"
#include "qbio.h"
#include "util.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

// we now need a separate set of callbacks for writing inspect files,
// and for decoding input soundfiles. this is because inspect files
// now need to be writable on a per-span basis, rather than in one long
// piece from a buffer, so we need the ability to append to an already
// open inspect file. meanwhile, input files *have* to be read from a
// buffer, because they might not be sound files; they might be cycap
// files instead, in which case we'd just have to read them into a buffer anyway. --NO LONGER TRUE



// libsndfile callbacks for reading sound files (from a buffer):
static sf_count_t sfcb_decode_tell (void *priv);
static sf_count_t sfcb_decode_write (const void *q, sf_count_t count, void *priv);
static sf_count_t sfcb_decode_read (void *q, sf_count_t count, void *priv);
static sf_count_t sfcb_decode_seek (sf_count_t offset, int whence, void *priv);
static sf_count_t sfcb_decode_get_filelen (void *priv);

static char *shorten_filename (char *fn);

//static qb_err_t sfcb_inspect_open_if_needed (qb_libsndfile_inspect_privdata_t *sfd);


static char *shorten_filename (char *fn) {
  size_t n, len;
  len = strlen(fn);
  for (n = len; n > 0; n--) {
    if (QB_PATHSEP == fn[n-1]) {
      return fn + n;
    }
  }
  return fn;
}





qb_err_t qb_decode_audio (u8_t *ipfile_buf,
                          s64_t ipfile_buflen,
                          char *ipfile_name,
                          u8_t use_right_channel,
                          qb_pcm_t *pcm,
                          u8_t dp) { // display_progress

  qb_err_t e;
  s64_t i;
  qb_libsndfile_decode_privdata_t sfd;
  SNDFILE *sndfile;
  int sr;
  SF_VIRTUAL_IO sf_virt;
  SF_INFO sf_info;
  int sf_fmt;
  sf_count_t r;
  float fbuf[2];
  char *name;
  
  memset(pcm, 0, sizeof(qb_pcm_t));
  e = QB_E_OK;
  
  memset(&sf_info, 0, sizeof(SF_INFO));
  memset(&sfd,     0, sizeof(qb_libsndfile_decode_privdata_t));
  memset(&sf_virt, 0, sizeof(SF_VIRTUAL_IO));
  
  // establish callbacks
  sf_virt.get_filelen = sfcb_decode_get_filelen;
  sf_virt.seek        = sfcb_decode_seek;
  sf_virt.read        = sfcb_decode_read;
  sf_virt.write       = sfcb_decode_write;
  sf_virt.tell        = sfcb_decode_tell;
  
  // stash privdata
  sfd.len = ipfile_buflen;
  sfd.pos = 0;
  sfd.buf = ipfile_buf;
  
//printf("priv = %p\n", &sfd);
//printf("sfd.len = %ld\n", sfd.len);
  
  sndfile = sf_open_virtual (&sf_virt, SFM_READ, &sf_info, &sfd);
  
  if (NULL == sndfile) {
    fprintf(QB_ERR, "B: %s: sf_open_virtual failed [%s]\n",
                    QB_FUNC_M, sf_strerror(sndfile));
    return QB_E_SNDFILE_OPEN;
  }
  
  do {

    sr     = sf_info.samplerate;
    sf_fmt = sf_info.format;
    
    name = shorten_filename(ipfile_name);

    e = qb_test_rate_legal (sr);
    if ( QB_E_OK != e ) {
      fprintf(QB_ERR, "E: File has illegal sample rate (%d). Aborting.\n", sr);
      break;
    }
    
    pcm->rate = 0x7fffffff & sr;
   
    if (sf_info.channels > 2) {
      fprintf(QB_ERR, "E: File has number of channels > 2 (%d). Aborting.\n",
              sf_info.channels);
      e = QB_E_NUM_CHANNELS;
      break;
    }
    
    if (sf_info.channels < 1) {
      fprintf(QB_ERR, "E: File has number of channels < 1 (%d). Aborting.\n",
              sf_info.channels);
      e = QB_E_NUM_CHANNELS;
      break;
    }
    
    if (use_right_channel && (sf_info.channels != 2)) {
      fprintf(QB_ERR, "E: Right channel selected on mono input file. Aborting.\n");
      e = QB_E_CLI_BAD_CHNL;
      break;
    }
    
    printf("Decoding \"%s\":\n  %s, %lld slices: ", //, fmt 0x%x
           name, (sf_info.channels == 2) ? "stereo" : "mono",
           (s64_t) sf_info.frames); //, sf_fmt);
    fflush(stdout); // MacOS
    
    pcm->channels      = 0x3 & sf_info.channels;
    pcm->is_big_endian = (sf_fmt & SF_ENDIAN_BIG) ? 1 : 0;
    pcm->is_float      = 0;
    pcm->is_signed     = 1;
    pcm->bytewidth     = 0;
    
    if (sf_info.frames > QB_MAX_IPFILE_SAMPLES) {
      fprintf(QB_ERR, "E: libsndfile produced too many frames. Aborting.\n");
      e = QB_E_SNDFILE_EXCESSIVE_FRAMES;
      break;
    }
    if (sf_info.frames < 1) {
      fprintf(QB_ERR, "E: libsndfile produced no frames. Aborting.\n");
      e = QB_E_SNDFILE_NO_FRAMES;
      break;
    }
    
    pcm->buf = qb_malloc(sf_info.frames * sizeof(float));
    if (NULL == pcm->buf) {
      fprintf(QB_ERR, "E: Out of memory reading audio file.\n");
      e = QB_E_MALLOC;
      break;
    }
    /*
    if (2 == pcm->channels) {
      pcm->buf_R = qb_malloc(sf_info.frames * sizeof(double));
      if (NULL == pcm->buf_R) {
        fprintf(QB_ERR, "E: Out of memory reading audio file.\n");
        return QB_E_MALLOC;
      }
    }
    */
    pcm->buflen = 0x7fffffff & sf_info.frames;
    
    // FIXME: read data more than 2 values at a time?
    // might be faster to deinterleave it ourselves once we own it, because
    // libsndfile has to go through this silly callback mechanism
    
    qb_show_meter(dp);
    
  //if ( ! use_right_channel ) { exit(0); }
    
    for (i=0; i < sf_info.frames; i++) {
      // get one slice
      r = sf_readf_float (sndfile, fbuf, 1);
      if (0 == r) {
        fprintf(QB_ERR, "\nE: libsndfile's sf_readf_double() was truncated "
                        "(pos is 0x%llx, i=%lld) [%s]\n",
                        (s64_t) sfd.pos, i, sf_strerror(sndfile));
        e = QB_E_SNDFILE_TRUNC;
        qb_free(pcm->buf);
        pcm->buf = NULL;
        break;
      }
      //pcm->buf_L[i] = dbuf[0];
      //if (2 == pcm->channels) {
      //  pcm->buf_R[i] = dbuf[1];
      //}
      
      pcm->buf[i] = fbuf[use_right_channel];

      qb_update_meter (dp, i, sf_info.frames, 1.0f, 0);
        
    } // next slice
    
    qb_hide_meter(dp, 0);
    
    if (QB_E_OK == e) {
      //qb_hide_meter(dp, 0);
      printf("done.\n");
    }
    
  } while (0);
  
  sf_close(sndfile);
  sndfile = NULL;
  
  if (QB_E_OK != e) { return e; }
  
#if defined QB_VECTORS_GCC_CLANG || defined QB_VECTORS_MSVC_AVX2
  if (QB_E_OK == e) {
    e = qb_vec_buf_init (&(pcm->vbuf),
                         QB_VECTYPE_FLOAT,
                         pcm->buf,
                         pcm->buflen,
                         1000, // 207
                         1000,
                         dp,
                         "  ");
  }
#endif

  if (QB_E_OK != e) { // error, clean up
    qb_finish_pcm (pcm);
  }
  
  return e;

}




// callbacks for INPUT FILE DECODE (sfcb_decode_...)
static sf_count_t sfcb_decode_get_filelen (void *priv) {
  qb_libsndfile_decode_privdata_t *sfd;
  sfd = (qb_libsndfile_decode_privdata_t *) priv;
//~ printf("priv = %p\n", sfd);
//~ printf("sfd.len = %ld\n", sfd->len);
//~ printf("sfcb_get_filelen() returns %lld\n", sfd->len);
  return sfd->len;
}

static sf_count_t sfcb_decode_seek (sf_count_t offset, int whence, void *priv) {
//~ printf("sfcb_seek(off=%lld, whence=%d)\n", (s64_t) offset, whence);
  qb_libsndfile_decode_privdata_t *sfd;
  sf_count_t pos;
  sfd = (qb_libsndfile_decode_privdata_t *) priv;
  pos = sfd->pos;
  if (whence == SEEK_SET) {
    pos = 0;
  } else if (whence == SEEK_END) {
    pos = sfd->len;
  }
  pos += offset;
  if (pos < 0) {
    // do something
    pos = 0;
  } else if (pos > sfd->len) { // it's allowed to be one past the end (EOF)
    // do something
    pos = sfd->len;
  }
  sfd->pos = pos;
  return pos;
}

static sf_count_t sfcb_decode_read (void *q, sf_count_t count, void *priv) {
  qb_libsndfile_decode_privdata_t *sfd;
  sf_count_t wanted;
  sfd = (qb_libsndfile_decode_privdata_t *) priv;
//~ printf("%lu\n", sizeof(sf_count_t)); exit(0);
//~ printf("sfd.buf = %p\n", sfd->buf);
//~ printf("read pos is %lld\n", (s64_t) sfd->pos);
//~ printf("sfcb_read(%lld)\n", (s64_t) count);
  wanted = count;
  if ((sfd->pos + wanted) > sfd->len) {
    wanted = sfd->len - sfd->pos; // truncate
  }
#ifdef QB_SANITY
  if (wanted < 0) {
    fprintf(QB_ERR, "\nsfcb_read: insane libsndfile somehow requested negative num bytes\n");
    return 0;
  }
#endif
  memcpy (q, sfd->buf + sfd->pos, wanted);
  sfd->pos += wanted;
  return wanted;
}

#define QB_SNDFILE_WRITE_ALLOC_DELTA 1000000

// not needed
static sf_count_t sfcb_decode_write (const void *q, sf_count_t count, void *priv) {
  
  qb_libsndfile_decode_privdata_t *ptr;
  u8_t *buf;
  s64_t new_alloc;
  u8_t *q8;
  
  q8 = (u8_t *) q;
  
  ptr = (qb_libsndfile_decode_privdata_t *) priv;
  
#ifdef QB_SANITY
  if (count > QB_MAX_IPFILE_SAMPLES) {
    fprintf(QB_ERR,
            "\nsfcb_write: insane libsndfile somehow tried to write %lld bytes\n",
            (s64_t) count);
    return 0;
  }
  if ((ptr->pos + count) > QB_MAX_IPFILE_SAMPLES) {
    fprintf(QB_ERR,
            "\nsfcb_write: tried to write more total audio frames than are allowed (%lld vs. %lld)\n",
            (s64_t) (ptr->len + count),
            QB_MAX_IPFILE_SAMPLES);
    return 0;
  }
#endif
  
  if ((ptr->pos + count) >= ptr->alloc) {
    new_alloc = ptr->alloc + count + QB_SNDFILE_WRITE_ALLOC_DELTA;
    buf = qb_realloc(ptr->buf, sizeof(s16_t) * new_alloc);
    if (NULL == buf) {
      fprintf(QB_ERR, "\nsfcb_write: realloc(%lld) failed\n", new_alloc);
      return 0;
    }
    ptr->buf = buf;
    ptr->alloc = new_alloc;
  }

  memcpy(ptr->buf + ptr->pos, q8, count);
  
  (ptr->pos) += count;
  
  if (ptr->pos >= ptr->len) {
    ptr->len = 1 + ptr->pos;
  }
  
  return count;
  
}

static sf_count_t sfcb_decode_tell (void *priv) {
//~ printf("sfcb_tell()\n");
  qb_libsndfile_decode_privdata_t *sfd;
  sfd = (qb_libsndfile_decode_privdata_t *) priv;
  return sfd->pos;
}






