#include "triangle.h"

#include <stdio.h>
#include <string.h>

#include "roomdata.h"
#include "parse.h"

static citerr_t bresenham_high (s16_t v1x,
                                s16_t v1y,
                                s16_t v2x,
                                s16_t v2y,
                                //vram_t *vram,  // NULL means no drawing; spans only
                                u8_t spans_only,
                                u8_t buf[SCRATCH_TRGL_BUF_HEIGHT * FRAMEBUF_WIDTH],
                                raster_span_t **spans_out,
                                u32_t *num_spans_out,
                                u32_t *alloced_spans_out);
                                
static citerr_t bresenham_low (s16_t v1x,
                               s16_t v1y,
                               s16_t v2x,
                               s16_t v2y,
                               //vram_t *vram,  // NULL means no drawing; spans only
                               u8_t spans_only,
                               u8_t buf[SCRATCH_TRGL_BUF_HEIGHT * FRAMEBUF_WIDTH],  // NULL means no drawing; spans only
                               raster_span_t **spans_out,
                               u32_t *num_spans_out,
                               u32_t *alloced_spans_out);

void triangle_init (triangle_t *t) {
  memset(t, 0, sizeof(triangle_t));
  t->fg_colour = -1;
  t->fg_plot_mode = -1;
}

citerr_t triangle_fix_coords (triangle_t *t) {
  u16_t x, y;
  x = to_16bit(t->v3_tmp.x_high, t->v3_tmp.x_low);
  y = to_16bit(t->v3_tmp.y_high, t->v3_tmp.y_low);
  // sanity
  if (y < 32) {
    printf("ERROR: triangle_fix_coords(): y < 32, underflow (%d)\n", y);
    return CE_TRIANGLE_Y_UNDERFLOW;
  }
  y -= 32; // compensate for nonzero graphical origin
  t->v3.x = 0xff & (x / 8); // FIXME: use floating point division instead? will MOS do this?
  t->v3.y = 0xff & (y / 4);
  t->v1.y = 175 - t->v1.y;
  t->v2.y = 175 - t->v2.y;
  t->v3.y = 175 - t->v3.y;
  return CE_OK;
}

void propagate_triangle_colours_and_plot_modes (triangle_t triangles [MAX_TRIANGLES],
                                                s8_t num_triangles) {
  s8_t cwc, cwpm, i;
  cwc = 0;
  cwpm = 0;
  for (i=0; i < num_triangles; i++) {
    triangle_t *t;
    t = triangles + i;
    if (t->fg_colour == -1) {
      t->fg_colour = cwc;
    } else {
      cwc = t->fg_colour;
    }
    if (t->fg_plot_mode == -1) {
      t->fg_plot_mode = cwpm;
    } else {
      cwpm = t->fg_plot_mode;
    }
  }
}

// wikipedia
/*
citerr_t bresenham  (s16_t x0,
                     s16_t y0,
                     s16_t x1,
                     s16_t y1,
                     u8_t r,
                     u8_t g,
                     u8_t b,
                     u8_t c,
                     vram_t *vram) {
                           
  s16_t dx, sx, dy, sy, err, e2;
  citerr_t e;
  
  dx = (x1 > x0) ? (x1 - x0) : (x0 - x1);
  sx = (x1 > x0) ? 1 : -1;
  dy = (y1 > y0) ? (y0 - y1) : (y1 - y0);
  sy = (y0 < y1) ? 1 : -1;
  err = dx + dy;
  
  while ( 1 ) {
    e = vram_linplot (vram, x0, y0, r, g, b, c);
    if (CE_OK != e) { return e; }
    if ((x0 == x1) && (y0 == y1)) {
      break;
    }
    e2 = 2 * err;
    if (e2 >= dy) { // e_xy+e_x > 0
      err += dy;
      x0 += sx;
    }
    if (e2 <= dx) { // e_xy+e_y < 0
      err += dx;
      y0 += sy;
    }
  }
  return CE_OK;
}
*/



// also wikipedia
citerr_t bresenham(s16_t v1x,
                   s16_t v1y,
                   s16_t v2x,
                   s16_t v2y,
                   //vram_t *vram, // NULL means no drawing; spans only
                   u8_t spans_only,
                   u8_t buf[SCRATCH_TRGL_BUF_HEIGHT * FRAMEBUF_WIDTH],
                   raster_span_t **spans_out,
                   u32_t *num_spans_out,
                   u32_t *alloced_spans_out) {
  
  s16_t abs_x, abs_y;
  citerr_t e;
  
  abs_y = (v2y > v1y) ? (v2y - v1y) : (v1y - v2y);
  abs_x = (v2x > v1x) ? (v2x - v1x) : (v1x - v2x);
  
  *num_spans_out = 0;
  
  // no -- don't do this!
  //~ if ( ! spans_only ) {
    //~ memset(buf, 0, SCRATCH_TRGL_BUF_HEIGHT * FRAMEBUF_WIDTH);
  //~ }
  
  if (abs_y < abs_x) {
    if (v1x > v2x) {
      e = bresenham_low (v2x, v2y, v1x, v1y, spans_only, buf, spans_out, num_spans_out, alloced_spans_out);
    } else {
      e = bresenham_low (v1x, v1y, v2x, v2y, spans_only, buf, spans_out, num_spans_out, alloced_spans_out);
    }
  } else {
    if (v1y > v2y) {
      e = bresenham_high (v2x, v2y, v1x, v1y, spans_only, buf, spans_out, num_spans_out, alloced_spans_out);
    } else {
      e = bresenham_high (v1x, v1y, v2x, v2y, spans_only, buf, spans_out, num_spans_out, alloced_spans_out);
    }
  }
  
  return e;
  
}

static citerr_t bresenham_low (s16_t v1x,
                               s16_t v1y,
                               s16_t v2x,
                               s16_t v2y,
                               //vram_t *vram,  // NULL means no drawing; spans only
                               u8_t spans_only,
                               u8_t buf[SCRATCH_TRGL_BUF_HEIGHT * FRAMEBUF_WIDTH],  // NULL means no drawing; spans only
                               raster_span_t **spans_out,
                               u32_t *num_spans_out,
                               u32_t *alloced_spans_out) {
  
  s16_t dy, dx, d, yi, y, x;
  u8_t new_span;
  raster_span_t span;
  raster_span_t *next_span;
  citerr_t e;
  
//~ printf("low\n");
  
  dx = v2x - v1x;
  dy = v2y - v1y;
  
  yi = 1;
  if (dy < 0) {
    yi = -1;
    dy = -dy;
  }
  d = (2 * dy) - dx;
  y = v1y;
  
  span.steep = 0;
  new_span = 1;

  for (x = v1x; x <= v2x; x++) {
  
    //u32_t off;
    
    //off = (((FRAMEBUF_WIDTH * y) + x)) + 0;
  
//~ printf("x=%d ", x);

    //plot(x, y)
    if ( ! spans_only ) { //vram) {
      //e = vram_linplot (vram, x, y, 0xff, 0xff, 0xff, 0x3);
      // reserve 16 rows at the top to make room for negative values
      buf[(FRAMEBUF_WIDTH * (y + (SCRATCH_TRGL_BUF_MARGIN / 2))) + x] = 255; // 8 bpp now
      //if (CE_OK != e) { return e; }
    }
    //~ (*total_pixels_out)++;
    if (new_span) {
      new_span = 0;
      span.start_x = x;
      span.start_y = y;
      span.len = 1;
    } else {
      span.len++;
    }
    if (d > 0) {
      // end of span
      new_span = 1;
      next_span = NULL;
      e = get_next_span(num_spans_out,
                        alloced_spans_out,
                        spans_out,
                        &next_span);
      if (CE_OK != e) { return e; }
//~ printf("get_next_span()\n");
      *next_span = span; // copy working span to list
      y += yi;
      d += (2 * (dy - dx));
    } else {
      d += (2 * dy);
    }
  }
  
  // end of final span
  e = get_next_span(num_spans_out,
                    alloced_spans_out,
                    spans_out,
                    &next_span);
  if (CE_OK != e) { return e; }
  *next_span = span; // copy final span to list
  
  return CE_OK;
  
}


static citerr_t bresenham_high (s16_t v1x,
                                s16_t v1y,
                                s16_t v2x,
                                s16_t v2y,
                                //vram_t *vram,  // NULL means no drawing; spans only
                                u8_t spans_only,
                                u8_t buf[SCRATCH_TRGL_BUF_HEIGHT * FRAMEBUF_WIDTH],
                                raster_span_t **spans_out,
                                u32_t *num_spans_out,
                                u32_t *alloced_spans_out) {

  s16_t dx, dy, xi, x, y, d;
  u8_t new_span;
  raster_span_t span;
  raster_span_t *next_span;
  citerr_t e;
  
//~ printf("high\n");

  dx = v2x - v1x;
  dy = v2y - v1y;
  xi = 1;
  
  if (dx < 0) {
    xi = -1;
    dx = -dx;
  }
  d = (2 * dx) - dy;
  x = v1x;
  
  span.steep = 1;
  new_span = 1;

  for (y = v1y; y <= v2y; y++) {
    if ( ! spans_only ) { //vram) {
      //e = vram_linplot (vram, x, y, 0xff, 0xff, 0xff, 0x3);
      // reserve 16 rows at the top to make room for negative values
      buf[(FRAMEBUF_WIDTH * (y + (SCRATCH_TRGL_BUF_MARGIN / 2))) + x] = 255; // 8 bpp now
      //if (CE_OK != e) { return e; }
    }
    if (new_span) {
      new_span = 0;
      span.start_x = x;
      span.start_y = y;
      span.len = 1;
    } else {
      span.len++;
    }
    if (d > 0) {
      // end of span
      new_span = 1;
      next_span = NULL;
      e = get_next_span(num_spans_out,
                        alloced_spans_out,
                        spans_out,
                        &next_span);
      if (CE_OK != e) { return e; }
      *next_span = span; // copy working span to list
      x += xi;
      d += (2 * (dx - dy));
    } else {
      d = d + (2 * dx);
    }
  }
  
  // end of final span
  e = get_next_span(num_spans_out,
                    alloced_spans_out,
                    spans_out,
                    &next_span);
  if (CE_OK != e) { return e; }
  *next_span = span; // copy final span to list
  
//~ printf("final num_spans = %u\n", *num_spans_out);
  
  return CE_OK;
  
}
