//#include "crunpack.h"
#include "tiledata.h"

#include <stdio.h>
#include <string.h>

void tile_slot_init (tile_slot_t *slot) {
  memset(slot, 0, sizeof(tile_slot_t));
}

void tile_data_init (tile_data_t *tiledata,
                     u8_t t955[TILEDATA_LEN],
                     u8_t t600[T_600_LEN]) {
                     
  tile_data_reset(tiledata);
  tile_data_wipe400(tiledata);
  memcpy(tiledata->t955, t955, TILEDATA_LEN);
  memcpy(tiledata->t600, t600, T_600_LEN);
  
}

void tile_data_reset (tile_data_t *tiledata) {
  tiledata->ptr_L = 0xff & TILEDATA_START;   // ptr as seen by 6502 code
  tiledata->ptr_H = 0xff & (TILEDATA_START >> 8);
}

citerr_t tile_data_advance (tile_data_t *tiledata, u8_t num_bytes, u8_t *carry_out) {
  u8_t carry;
  u16_t i;
  carry=0;
  i = tiledata->ptr_L;
//printf("num_bytes = %x\n", num_bytes);
//printf("pre-advance: ptr_L = %x\n", tiledata->ptr_L);
  i += num_bytes;
  if (i > 255) {
    carry = 1;
    i -= 256;
    if (tiledata->ptr_H == 0xff) {
      printf("ERROR: tile_data_advance(): ptr_H overflow\n");
      return CE_TILE_DATA_PTR_H_OVERFLOW;
    }
    tiledata->ptr_H++;
  }
  tiledata->ptr_L = 0xff & i;
//printf("advance: ptr_L = %x\n", tiledata->ptr_L);
  *carry_out = carry;
  return CE_OK;
}

citerr_t tile_data_regress (tile_data_t *tiledata, u8_t *carry_out) {
  // this is used to make the pointer move back to the previous tile
  // the distance to regress comes from the tile data stream ...
  u8_t carry;
  u8_t x;
  s16_t i;
  citerr_t e;
  carry = 0;
  //s = sprintf("329c regress_tileptr: 0x%x . 0x", this.getptr());
  e = tile_data_read (tiledata, &x);
  if (CE_OK != e) { return e; }
  i = tiledata->ptr_L;
  i -= x;
  tiledata->ptr_L = 0xff & i;
  if (i < 0) {
    if (tiledata->ptr_H == 0) {
      printf("ERROR: tile_data_regress(): ptr_H negative\n");
      return CE_TILE_DATA_PTR_H_NEGATIVE;
    }
    tiledata->ptr_H--;
    carry = 1;
  }
  //trace(s . sprintf("%x", this.getptr()));
  //trace("32a0 byte to subtract from accumulator=".sprintf("0x%x", x));
  *carry_out = carry;
  return CE_OK;
}

citerr_t tile_data_read (tile_data_t *tiledata, u8_t *out) {
  return tile_data_offread(tiledata, 0, out);
}

citerr_t tile_data_offread (tile_data_t *tiledata, u8_t off, u8_t *out) {
  
  u16_t p;
  
  p = tile_data_getptr(tiledata);
  
  if ((((u32_t) p) + off) > 0xffff) {
    printf("ERROR: tile_data_offread(): offset (0x%x) overflows ptr (0x%x)\n",
           off, p);
    return CE_TILE_DATA_OFFREAD_P_OVERFLOW;
  }
  
  p += off;
  
  return tile_data_extread (tiledata, p, out);

}

citerr_t tile_data_extread (tile_data_t *tiledata, u16_t ptr, u8_t *out) {
  
  u8_t *buf;
  s32_t q;
  u16_t len;
  
  if (tile_data_getextsrc(ptr) == TILE_DATASRC_400) {
    q = ptr - T_400_PTR_START;
    len = T_400_LEN;
    buf = tiledata->t400;
  } else if (tile_data_getextsrc(ptr) == TILE_DATASRC_600) {
    q = ptr - T_600_PTR_START;
    len = T_600_LEN;
    buf = tiledata->t600;
  } else {
    q = ptr - TILEDATA_START;
    len = TILEDATA_LEN;
    buf = tiledata->t955;
  }
  
  if (q < 0) {
    printf("ERROR: bad TileData read attempt (underflow @ 0x%x)\n", ptr);
    return CE_TILEREAD_UNDERFLOW;
  } else if (q >= len) {
    printf("ERROR: bad TileData read attempt (overflow @ 0x%x)\n", ptr);
    return CE_TILEREAD_OVERFLOW;
  }
  
  *out = buf[q];
  
  return CE_OK;
  
}


citerr_t tile_data_write (tile_data_t *tiledata, u16_t ptr, u8_t c) {
  s32_t p;
  // writes only allowed to t400
  p = ptr - T_400_PTR_START;
  if (p < 0) {
    printf("ERROR: bad tile_data_write() attempt (underflow @ 0x%x)\n", ptr);
    return CE_T400_WRITE_UNDERFLOW;
  } else if (p >= T_400_LEN) {
    printf("ERROR: bad tile_data_write() attempt (overflow @ 0x%x, t400 len is 0x%x)\n", ptr, T_400_LEN);
    return CE_T400_WRITE_OVERFLOW;
  }
  tiledata->t400[p] = c;
  return CE_OK;
}


citerr_t tile_data_getsrc (tile_data_t *tiledata) {
  return tile_data_getextsrc (tile_data_getptr (tiledata));
}

u8_t tile_data_getextsrc (u16_t ptr) {
  if ((ptr >= T_400_PTR_START) && (ptr < (T_400_PTR_START + T_400_LEN))) {
    return TILE_DATASRC_400;
  } else if ((ptr >= T_600_PTR_START) && (ptr < (T_600_PTR_START + T_600_LEN))) {
    return TILE_DATASRC_600;
  } else {
    return TILE_DATASRC_TILEDATA;
  }
}

void tile_data_wipe400 (tile_data_t *tiledata) {
  memset(tiledata->t400, 0x41, T_400_LEN);
}

void tile_data_setptr (tile_data_t *tiledata, u16_t ptr) {
  tiledata->ptr_L = ptr & 0xff;
  tiledata->ptr_H = (ptr >> 8) & 0xff;
}

u16_t tile_data_getptr (tile_data_t *tiledata) {
  return to_16bit (tiledata->ptr_H, tiledata->ptr_L);
}
