#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include "krunpack.h"
#include "citconf.h"
#include "cit_io.h"
#include "world.h"
#include "roomdata.h"
#include "parse.h"
#include "tiledata.h"
#include "vram.h"
#include "citpng.h"

#define CLI_MODE_OPTIONS_STARTED       1
#define CLI_MODE_ROOM_ID               2
#define CLI_MODE_HAVE_ROOM_ID          3
#define CLI_MODE_HAVE_VISIBLE_FILENAME 4
#define CLI_MODE_DONE                  5
#define CLI_MODE_ITEM_TILE_ID          6
#define CLI_MODE_CRYSTALS              7

// [ ] pixel_base_pairs, animation_speeds: redefine these to include sizes
// [ ] do something about animation_speeds being passed around needlessly all over the place -- just pass NULL?

static citerr_t parse_cli (int argc,
                           char *argv[],
                           citadel_config_t *cc,
                           char **citam_out,              // uses memory from argv
                           char **op_filename_out,        // uses memory from argv
                           char **collision_filename_out, // uses memory from argv
                           room_id_t *room_id_out);
                           
static citerr_t parse_u8 (char *s, u8_t *out);
static void usage (char *argv0);

static void print_config (room_id_t *room_id,
                          char *op_filename,
                          char *collision_filename,
                          citadel_config_t *cc);
                          
static citerr_t citadel_main  (char *citam,
                               u8_t **citam_contents_out,
                               room_id_t *room_id,
                               citadel_config_t *cc,
                               char *output_filename,
                               char *collision_filename);
                               
static citerr_t find_magic  (u8_t *haystack,
                             size_t haystack_len,
                             u8_t *needle,
                             size_t needle_len,
                             size_t *result_out);
                             
static citerr_t walk_chain  (room_data_t *roomdata,
                             tile_data_t *tiledata,
                             room_id_t *room_id,
                             citadel_config_t *config,
                             world_data_t *world,
                             vram_t *vram,
                             s16_t *sm_tactile_x_out,
                             s16_t *sm_tactile_y_out,
                             s16_t *flask_x_out,
                             s16_t *flask_y_out);
                             
                             
static citerr_t gfx_buffer_to_png_for_room (u8_t *gfx_buffer,
                                            char *png_output_filename,
                                            u8_t extended_screen,
                                            size_t *filesize_out);

int main (int argc, char *argv[]) {
  citerr_t e;
  citadel_config_t cc;
  room_id_t rid;
  char *citam;
  char *op_filename;
  char *collision_filename;
  u8_t *citam_contents;
  citam = NULL;
  op_filename = NULL;
  collision_filename = NULL;
  e = parse_cli (argc,
                 argv,
                 &cc,
                 &citam,
                 &op_filename,
                 &collision_filename,
                 &rid);
  if (CE_OK != e) { return e; }
  print_config(&rid, op_filename, collision_filename, &cc);
  e = citadel_main(citam,
                   &citam_contents,
                   &rid,
                   &cc,
                   op_filename,
                   collision_filename);
  if (NULL != citam_contents) {
    free(citam_contents);
  }
  return e;
//#warning main() returning CE_OK
  //return CE_OK; // lul
}

static citerr_t parse_cli (int argc,
                           char *argv[],
                           citadel_config_t *cc,
                           char **citam_out,              // uses memory from argv
                           char **op_filename_out,        // uses memory from argv
                           char **collision_filename_out, // uses memory from argv
                           room_id_t *room_id_out) {

  u8_t mode;
  // u8_t have_a_cry;
  int i;
  citerr_t e;

  mode = CLI_MODE_OPTIONS_STARTED;
  // have_a_cry = 0;
  citadel_config_init(cc);
  room_id_init(room_id_out);
  
  printf("\n" CIT_VERSION "\n\n");
  
  for (i=1 ; i < argc; i++) {
  
    char *a;
    size_t z;
    s8_t cry;
    u8_t b;
  
    a = argv[i];
    
    // FIXME: block repeated options
    
    switch (mode) {
      case CLI_MODE_OPTIONS_STARTED:
        if (a[0] == '+') {
          if ( ! strcmp (a, "+wd") ) {
            cc->well_drained = 1;
          } else if ( ! strcmp (a, "+sp") ) {
            cc->star_port_destroyed = 1;
          } else if ( ! strcmp (a, "+ic") ) {
            cc->ice_crystal = 1;
          } else if ( ! strcmp (a, "+to") ) {
            cc->triangle_outlines_only = 1;
#ifdef EXTENDED_HEIGHT
          } else if ( ! strcmp (a, "+xs") ) {
            cc->extended_screen = 1;
#endif
          } else if ( ! strcmp (a, "+ud") ) {
            if (cc->doors_unlocked != DOORS_LOCKED) {
              printf("ERROR: can only use one of +ud or +od.\n");
              return CE_UD_WITH_OD;
            }
            cc->doors_unlocked = DOORS_UNLOCKED;
          } else if ( ! strcmp (a, "+od") ) {
            if (cc->doors_unlocked != DOORS_LOCKED) {
              printf("ERROR: can only use one of +ud or +od.\n");
              return CE_UD_WITH_OD;
            }
            cc->doors_unlocked = DOORS_OPENED;
          } else if ( ! strcmp (a, "+it") ) {
            mode = CLI_MODE_ITEM_TILE_ID;
          } else if ( ! strcmp (a, "+cr") ) {
            mode = CLI_MODE_CRYSTALS;
          } else if ( ! strcmp (a, "+lo") ) {
            cc->lab_opened = 1;
          } else if ( ! strcmp (a, "+bl") ) {
            cc->battlements_lift_on = 1;
          } else {
            printf("ERROR: Unrecognised option: %s\n", a);
            usage(argv[0]);
            return CE_USAGE;
          }
          break;
        } //else {
          //mode = CLI_MODE_OPTIONS_ENDED;
          // *** FALL THROUGH ***
        //}
        // options finished
        // expect citam/citax
        *citam_out = a;
        mode = CLI_MODE_ROOM_ID;
        break;
      case CLI_MODE_ROOM_ID:
        // expect room ID
        e = parse_u8(a, &b);
        if (CE_OK != e) {
          printf("ERROR: Room ID is illegal (%s).\n", a);
          return CE_BAD_ROOM_ID;
        }
        room_id_out->original = b;
        mode = CLI_MODE_HAVE_ROOM_ID;
        break;
      case CLI_MODE_CRYSTALS:
        // len = strlen(a);
        for (z=0; z < strlen(a); z++) {
          cry = -1;
          switch (a[z]) {
            case '0': cry = 0; /*have_a_cry = 1;*/ break;
            case '1': cry = 1; /*have_a_cry = 1;*/ break;
            case '2': cry = 2; /*have_a_cry = 1;*/ break;
            case '3': cry = 3; /*have_a_cry = 1;*/ break;
            case '4': cry = 4; /*have_a_cry = 1;*/ break;
            default:
              printf("ERROR: Crystal ID list format is wrong (should be e.g. 01234 or 034 or 1, etc.)\n");
              return CE_USAGE;
          }
          if (cc->crystals[cry] != 0) {
            printf("ERROR: Duplicated crystal ID (%u).\n", cry);
            return CE_USAGE;
          }
          cc->crystals[cry] = 1;
        }
        mode = CLI_MODE_OPTIONS_STARTED;
        break;
      case CLI_MODE_HAVE_ROOM_ID:
        *op_filename_out = a;
        mode = CLI_MODE_HAVE_VISIBLE_FILENAME;
        break;
      case CLI_MODE_HAVE_VISIBLE_FILENAME:
        *collision_filename_out = a;
        mode = CLI_MODE_DONE;
        break;
      case CLI_MODE_ITEM_TILE_ID:
        e = parse_u8(a, &b);
        if (CE_OK != e) {
          printf("ERROR: Item tile ID is illegal (%s).\n", a);
          return CE_BAD_ITEM_TILE_ID;
        }
        if (b > TID_MAX) {
          printf("ERROR: Item tile ID is illegal (%u, max. %u).\n", b, TID_MAX);
          return CE_BAD_ITEM_TILE_ID;
        }
        cc->item_tile_id = b;
        mode = CLI_MODE_OPTIONS_STARTED;
        break;
      default:
        usage(argv[0]);
        return CE_USAGE;
    }
    
  } // next argv
  
  if ((mode != CLI_MODE_DONE) && (mode != CLI_MODE_HAVE_VISIBLE_FILENAME)) {
    usage(argv[0]);
    return CE_USAGE;
  }
  
  return CE_OK;
  
}


static citerr_t parse_u8 (char *s, u8_t *out) {
  int i;
  u32_t u;
  size_t len, z;
  len = strlen(s);
  for (z=0; z < len; z++) {
    if ( ! isdigit(s[z]) ) {
      return CE_U8_PARSE_NON_NUMERIC;
    }
  }
  i = CIT_SSCANF(s, "%u", &u);
  if (i != 1) {
    return CE_U8_PARSE_SSCANF;
  }
  if (u > 0xff) {
    return CE_U8_PARSE_TOO_BIG;
  }
  *out = u & 0xff;
//printf("parse_u8: \"%s\" -> %u\n", s, *out);
  return CE_OK;
}


static void usage (char *argv0) {
  printf("Usage; [...] = optional; <...> = mandatory\n\n"
         "%s [switches] <CITAM file> <room ID> <PNG> [collision PNG]\n\n"
         "Switches:\n"
         "  +wd             The Well has been drained\n"
         "  +bl             Central Tower battlements lift has been activated\n"
         "  +sp             Star Port has been destroyed\n"
         "  +ic             ice crystal is in use\n"
         "  +ud, +od        unlock doors, open doors (mutually exclusive)\n"
         "  +lo             The Lab has been opened\n"
         "  +it <tile id>   draw item with tile ID <id> on item pad, if present\n"
         "  +cr <cry. ids>  specify list of IDs (0-4) of crystals returned (e.g. \"034\")\n"
         "  +to             draw PLOT 85 triangles as outlines only\n"
         "  +xs             extend screen height to 192 pixels, show ROM overwrites\n"
         "\n", argv0);
}


static void print_config (room_id_t *room_id,
                          char *op_filename,
                          char *collision_filename,
                          citadel_config_t *cc) {
                          
  u8_t have_a_cry, i;
  
  have_a_cry = 0;
                          
  printf("Configuration:\n");
  printf("  + Room ID is %u (0x%x).\n", room_id->original, room_id->original);
  printf("  + Writing visible output to \"%s\".\n", op_filename);
  
  if (collision_filename != NULL) {
    printf("  + Writing collision output to \"%s\".\n", collision_filename);
  }
  
  if (cc->well_drained)        { printf("  + Draining The Well.\n"); }
  if (cc->star_port_destroyed) { printf("  + Destroying the Star Port.\n"); }
  if (cc->ice_crystal)         { printf("  + Have ice crystal.\n"); }
  if (cc->doors_unlocked == DOORS_UNLOCKED) {
    printf("  + Doors are unlocked (visible but permeable).\n");
  } else if (cc->doors_unlocked == DOORS_OPENED) {
    printf("  + Doors are all open.\n");
  }
  for (i=0; i < 5; i++) {
    if (1 == cc->crystals[i]) {
      have_a_cry = 1;
    }
  }
  if (have_a_cry) {
    printf("  + Crystals: ");
    for (i=0; i < 5; i++) {
      if (1 == cc->crystals[i]) {
        printf("%u ", i);
      }
    }
    printf("\n");
  }
  if (cc->lab_opened)          { printf("  + Lab is open.\n"); }
  if (cc->battlements_lift_on) { printf("  + Battlements lift is active (switch flipped).\n"); }
  if (cc->item_tile_id > -1)   { printf("  + Room item tile ID is %u.\n", cc->item_tile_id); }
}

#define FLASKS_ROOM_TABLE_BACKTRACK_FROM_TILE_DATA 0xa5
#define FLASKS_POSITIONS_OFFSET_FROM_FLASKS_ROOMS  0x40

static citerr_t citadel_main  (char *citam,
                               u8_t **citam_contents_out, // out (for free() only)
                               room_id_t *room_id,
                               citadel_config_t *cc,
                               char *output_filename,
                               char *collision_filename) {
  
  citerr_t e;
  size_t buflen;
  u8_t *buf;
  u16_t citam_len, rdpos, tdpos;
  u16_t pxpairs_pos, animspeeds_pos, globroomnames_pos;
  u16_t flasks_rooms_pos, flasks_positions_pos;
  u16_t t600pos;
  u8_t *magic;
  world_data_t world;
  u8_t i;
  u32_t w;
  size_t z;
  s16_t sm_tactile_x, sm_tactile_y;
  s16_t flask_x, flask_y;
  room_data_t roomdata;
  tile_data_t tiledata;
  vram_t vram;
  size_t vis_filesize, coll_filesize;

  printf("\nInput file:\n");

  *citam_contents_out = NULL;
  
  t600pos = 0; // FIXME !!

  e = cit_read_file(citam, citam_contents_out, &buflen);
  
  if (CE_OK != e) {
    printf("ERROR: Could not load file \"%s\"\n", citam);
    return e; //CE_LOAD;
  }

  buf = *citam_contents_out;
  
  if (buflen > 0xffff) {
    printf("ERROR: Loaded file is too large (%zu bytes, must be < 65535).\n", buflen);
    return CE_CITAM_TOO_BIG;
  }
  if (buflen == 0) {
    printf("ERROR: Loaded file is empty.\n");
    return CE_CITAM_ZL;
  }
  
  citam_len = 0xffff & buflen;
  
  magic = PRE_ROOM_DATA_MAGIC;
  e = find_magic (buf, buflen, magic, L_PRE_ROOM_DATA_MAGIC, &z);
  if (CE_OK != e) {
    printf("ERROR: Could not find pre-room-data magic.\n");
    return CE_ROOM_MAGIC;
  }
  printf("  + Found pre-room magic at 0x%zx; room data follows at 0x%zx.\n",
         z, z + L_PRE_ROOM_DATA_MAGIC);
  z += L_PRE_ROOM_DATA_MAGIC;
  if (z > citam_len) {
    printf("ERROR: room data pos > citam len (%zu vs. %u)\n", z, citam_len);
    return CE_TRUNC_ROOM_DATA;
  }
  rdpos = 0xffff & z;

  magic = PRE_TILE_DATA_MAGIC;
  e = find_magic (buf, buflen, magic, L_PRE_TILE_DATA_MAGIC, &z);
  if (CE_OK != e) {
    printf("ERROR: Could not find pre-tile-data magic.\n");
    return CE_TILE_MAGIC;
  }
  printf("  + Found pre-tile magic at 0x%zx; tile data follows at 0x%zx.\n",
         z, z + L_PRE_TILE_DATA_MAGIC);
  z += L_PRE_TILE_DATA_MAGIC;
  if (z > citam_len) {
    printf("ERROR: tile data pos > citam len (%zu vs. %u)\n", z, citam_len);
    return CE_TRUNC_TILE_DATA;
  }
  tdpos = 0xffff & z;

  magic = PRE_PXPAIRS_ANIMSPEEDS_MAGIC;
  e = find_magic (buf, buflen, magic, L_PRE_PXPAIRS_ANIMSPEEDS_MAGIC, &z);
  if (CE_OK != e) {
    printf("ERROR: Could not find pixel pairs / animation speeds magic.\n");
    return CE_PXPAIRS_ANIMSPEEDS_MAGIC;
  }
  printf("  + Found pre-pxpairs magic at 0x%zx; pxpairs data follows at 0x%zx.\n",
         z, z + L_PRE_PXPAIRS_ANIMSPEEDS_MAGIC);
  z += L_PRE_PXPAIRS_ANIMSPEEDS_MAGIC;
  if (z > citam_len) {
    printf("ERROR: pxpairs data pos > citam len (%zu vs. %u)\n", z, citam_len);
    return CE_TRUNC_PXPAIRS_DATA;
  }
  pxpairs_pos = 0xffff & z;

  z += 16;
  
  if (z > citam_len) {
    printf("ERROR: animspeeds data pos > citam len (%zu vs. %u)\n", z, citam_len);
    return CE_TRUNC_ANIMSPEEDS_DATA;
  }
  animspeeds_pos = 0xffff & z;

  magic = POST_GLOBAL_ROOMNAMES_MAGIC;
  e = find_magic (buf, buflen, magic, L_POST_GLOBAL_ROOMNAMES_MAGIC, &z);
  if (CE_OK != e) {
    printf("ERROR: Could not post-global room name table magic.\n");
    return CE_GLOBAL_ROOMNAMES_MAGIC;
  }
  if (z < WORLD_GLOBAL_ROOM_TBL_LEN) {
    printf("ERROR: global room name data expected before start of file (at %zu - %u)\n",
           z, WORLD_GLOBAL_ROOM_TBL_LEN);
    return CE_GLOBAL_ROOMNAMES_UNDERFLOW;
  }
  printf("  + Found post-global room name table magic at 0x%zx; room name data precedes it at 0x%zx.\n",
         z, z - WORLD_GLOBAL_ROOM_TBL_LEN);
  globroomnames_pos = 0xffff & (z - WORLD_GLOBAL_ROOM_TBL_LEN);

  if ((rdpos == 0xfa0) && (tdpos == 0x555)) {
    if (citam_len == CITAM_LEN) {
      printf("  + Original's CITAM detected (based on length of %u).\n", CITAM_LEN);
    } else if (citam_len == CITAX_LEN) {
      printf("  + Two-file version's CITAX detected (based on length of %u).\n", CITAM_LEN);
    } else {
      printf("  + Unknown data version detected (rogue file length of %u).\n", citam_len);
    }
  } else {
    printf("  + Unknown data version detected (strange tile data and/or room data positions).\n");
  }
  
  world_init(&world);

  printf("  + Pixel base pairs:\n    ");
  
  for (i=0; i < WORLD_NUM_PXPAIRS; i++) {
    u8_t j;
    j = buf[pxpairs_pos + i];
    world.pixel_base_pairs[i] = j;
    printf("%02x ", j);
  }
  
  printf("\n");
  printf("  + Animation speeds:\n    ");
  
  for (i=0; i < WORLD_NUM_ANIMSPEEDS; i++) {
    u8_t j;
    j = buf[animspeeds_pos + i];
    world.animation_speeds[i] = j;
    printf("%02x ", j);
  }
  
  printf("\n");
  
  //flasks_room_table_backtrack_from_tile_data = 0xa5;
  
  if (FLASKS_ROOM_TABLE_BACKTRACK_FROM_TILE_DATA > tdpos) {
    printf("ERROR: flasks room table expected before start of file (backtrack %u from %u)\n",
           FLASKS_ROOM_TABLE_BACKTRACK_FROM_TILE_DATA, tdpos);
    return CE_FLASKS_ROOMS_UNDERFLOW;
  }
  
  flasks_rooms_pos = (tdpos - FLASKS_ROOM_TABLE_BACKTRACK_FROM_TILE_DATA);
  
  printf("  + Assuming room flask table is %u bytes before tile data:\n    ",
         FLASKS_ROOM_TABLE_BACKTRACK_FROM_TILE_DATA);

  for (i=0; i < WORLD_NUM_FLASKS; i++) {
    u8_t j;
    j = buf[flasks_rooms_pos + i];
    world.flasks_room_ids[i] = j;
    printf("%02x ", j);
  }
  printf("\n");
  
  w = FLASKS_POSITIONS_OFFSET_FROM_FLASKS_ROOMS + (u32_t) flasks_rooms_pos;
  
  if (w >= citam_len) {
    printf("ERROR: flask positions table expected beyond end of file (%u vs. %u)\n", w, citam_len);
    return CE_TRUNC_FLASKS_POS;
  }
  
  flasks_positions_pos = 0xffff & w;
  
  printf("  + Assuming flask positions table is %u "
         "bytes after room flask table:\n    ",
         FLASKS_POSITIONS_OFFSET_FROM_FLASKS_ROOMS);

  for (i=0; i < 16; i++) {
    u8_t j;
    j = buf[flasks_positions_pos + i];
    world.flasks_positions[i] = j;
    printf("%02x ", j);
  }
  printf("\n");
  
  memcpy(world.room_name_page, buf + globroomnames_pos, WORLD_GLOBAL_ROOM_TBL_LEN);
  
  sm_tactile_x = -1;
  sm_tactile_y = -1;
  flask_x = -1;
  flask_y = -1;
  
  room_data_init (&roomdata, buf + rdpos, citam_len - rdpos);
  tile_data_init (&tiledata, buf + tdpos, buf + t600pos); // FIXME: possible oflow w/truncated file?
  vram_init (&vram, cc->extended_screen ? 192 : 176);
  
  // FIXME: make it return an error code
  e = walk_chain (&roomdata, //rooms,
                  &tiledata, //tld,
                  room_id, // potentially modified
                  cc,
                  &world,
                  &vram, //&vram,
                  &sm_tactile_x,       // out
                  &sm_tactile_y,       // out
                  &flask_x,            // out
                  &flask_y);           // out

  if (CE_OK != e) { //&& (CE_REDIR != e) ) {
    //printf("ERROR: walk_chain failed.\n");
    return e;
  }
  
  if (sm_tactile_x != -1) {
    printf("  + Room has a tactile at (%d, %d).\n", sm_tactile_x, sm_tactile_y);
  }
  
  
  e = gfx_buffer_to_png_for_room(vram.linvis,
                                 output_filename,
                                 cc->extended_screen,
                                 &vis_filesize);
  if (CE_OK != e) { return e; }
  coll_filesize = 0; // compiler is a moron
  if (NULL != collision_filename) {
    e = gfx_buffer_to_png_for_room(vram.lincol,
                                   collision_filename,
                                   cc->extended_screen,
                                   &coll_filesize);
    if (CE_OK != e) { return e; }
  }
  
  printf("  + Wrote \"%s\" (%zu bytes)", output_filename, vis_filesize);
  if (NULL != collision_filename) {
    printf(" and \"%s\" (%zu bytes)", collision_filename, coll_filesize);
  }
  printf(".\n");
  
  //gfx_crc(room_id, citvram.linvis, citvram.lincol);

  printf("Terminated normally.\n");
  
  
  return CE_OK;

}



static citerr_t find_magic  (u8_t *haystack,
                             size_t haystack_len,
                             u8_t *needle,
                             size_t needle_len,
                             size_t *result_out) {
                     
  // Boyer-Moore? Never heard of it
  
  size_t i;
  
  *result_out = 0;
  
  if (0 == needle_len) {
    printf("ERROR: bad find_magic() call: needle_len = 0\n");
    return CE_MAGIC_ZL_NEEDLE;
  }
  if (0 == haystack_len) {
    printf("ERROR: bad find_magic() call: haystack_len = 0\n");
    return CE_MAGIC_ZL_HAYSTACK;
  }
  if (needle_len > haystack_len) {
    printf("ERROR: find_magic(): needle_len(%zu) > haystack_len(%zu)",
           needle_len, haystack_len);
    return CE_MAGIC_SHORT_HAYSTACK;
  }
  
  for (i=0; i <= (haystack_len - needle_len); i++) {
    if (0 == memcmp(haystack + i, needle, needle_len)) {
      *result_out = i;
      return CE_OK;
    }
  }
  
  return CE_NO_MAGIC;
  
}



static citerr_t walk_chain  (room_data_t *roomdata,
                             tile_data_t *tiledata,
                             room_id_t *room_id,
                             citadel_config_t *config,
                             world_data_t *world,
                             vram_t *vram, //Vram cvr,
                             s16_t *sm_tactile_x_out,
                             s16_t *sm_tactile_y_out,
                             s16_t *flask_x_out,
                             s16_t *flask_y_out) {
  
  u8_t room_length;
  s16_t wanted_room_id;
  u8_t r;
  citerr_t e;
  
  *flask_x_out = -1;
  *flask_y_out = -1;
  room_length = 0;
  
//printf("room_id: orig = %d, redir = %d\n", room_id->original, room_id->redirected);
  
  wanted_room_id = room_id_fetch(room_id);
  
//printf("want[1] %d\n", wanted_room_id);

//printf("roomdata->pos = %x\n", roomdata->pos);
  
  for ( r=0;
        CE_OK == room_data_inbounds(roomdata); /*&& (room_length = room_data_read(roomdata, CE_TRUNC)*/ // /*&& ! (room_length & 0x80)*/
        r++ ) {
        
//printf("%x ", roomdata->pos);
//printf("rdt = %u\n", room_data_inbounds(roomdata));
//printf("want %d\n", wanted_room_id);
        
    u8_t redir;
    room_data_t room;
    
    e = room_data_read (roomdata, &room_length);
    if (CE_OK != e) { return CE_TRUNC; }
    
    // sanity
    if (0 == room_length) {
      printf("ERROR: bug or file corruption; zero room length!\n");
      return CE_ZERO_ROOM_LENGTH;
    }
    
//printf ("room_len = %x\n", room_length);
    
    // first byte is length of record
    
    if (wanted_room_id == r) {
    
      if (room_length == 1) {
        printf("ERROR: requested an undefined room\n");
        return CE_WALK_CHAIN_UNDEF_ROOM;
      }
      
      if (room_length == 2) {
      
        // special case: redirect; start again with room ID in following byte
        
        if (room_id->redirected != -1) {
          printf("ERROR: double room redirect\n");
          return CE_DOUBLE_ROOM_REDIRECT;
        }
        
        e = room_data_read(roomdata, &redir);
        if (CE_OK != e) { return CE_TRUNC_ROOM_LEN_2; }
        room_id->redirected = redir;
        
        printf("  + Redirect: new room ID is %d (0x%x)\n", room_id->redirected, room_id->redirected);
        
        // recurse ...
        (void) room_data_reset(roomdata);
        
        return walk_chain (roomdata,
                           tiledata,
                           room_id,
                           config,
                           world,
                           vram,
                           sm_tactile_x_out,
                           sm_tactile_y_out,
                           flask_x_out,
                           flask_y_out);
        
      } else {
      
        e = room_data_rewind(roomdata);
        if (CE_OK != e) { return e; }
        
        (void) room_data_init(&room, roomdata->buf + roomdata->pos + 1, room_length);
        
// 100:
//printf("roomdata->pos = %x\n", roomdata->pos);
//exit(0);
        
        e = parse_room(room_id->original,
                       &room,
                       tiledata,
                       config,
                       world,
                       flask_x_out,
                       flask_y_out,
                       vram,
                       sm_tactile_x_out,
                       sm_tactile_y_out);
                       
        if (CE_OK != e) { return e; }

      }
      
      return CE_OK;
      
    }
    
    // skip to next room
    e = room_data_advance (roomdata, room_length - 1);
    if (CE_OK != e) { return e; }
    
  }
  
  return CE_WALK_CHAIN_UNDEF_ROOM;
  
}


u16_t to_16bit (u8_t hi, u8_t lo) {
  u16_t _hi, _lo;
  _hi = hi;
  _lo = lo;
  return (0xff & _lo) | ((_hi << 8) & 0xff00);
}

static citerr_t gfx_buffer_to_png_for_room (u8_t *gfx_buffer,
                                            char *png_output_filename,
                                            u8_t extended_screen,
                                            size_t *filesize_out) {
                                     
  s32_t height;
  u16_t x,y;
  u8_t doubled[FRAMEBUF_HEIGHT * FRAMEBUF_WIDTH * 4 * 2];
  citerr_t e;
  //size_t filesize;

  height = extended_screen ? FRAMEBUF_MAX_HEIGHT : FRAMEBUF_MIN_HEIGHT;
     
  for (x = 0; x < FRAMEBUF_WIDTH*4; x += 4) { // step 4
    for (y = 0; y < height; y++) {
    
      u8_t r, g, b;
      
      r = gfx_buffer[0 + x + (FRAMEBUF_WIDTH * 4 * y)];
      g = gfx_buffer[1 + x + (FRAMEBUF_WIDTH * 4 * y)];
      b = gfx_buffer[2 + x + (FRAMEBUF_WIDTH * 4 * y)];
      // ignore alpha ...
      
      // one pixel in, two pixels out; double from 160 to 320
      doubled[0 + (2*x) + (FRAMEBUF_WIDTH * 4 * y * 2)] = r;
      doubled[1 + (2*x) + (FRAMEBUF_WIDTH * 4 * y * 2)] = g;
      doubled[2 + (2*x) + (FRAMEBUF_WIDTH * 4 * y * 2)] = b;
      doubled[3 + (2*x) + (FRAMEBUF_WIDTH * 4 * y * 2)] = 0xff;
      doubled[4 + (2*x) + (FRAMEBUF_WIDTH * 4 * y * 2)] = r;
      doubled[5 + (2*x) + (FRAMEBUF_WIDTH * 4 * y * 2)] = g;
      doubled[6 + (2*x) + (FRAMEBUF_WIDTH * 4 * y * 2)] = b;
      doubled[7 + (2*x) + (FRAMEBUF_WIDTH * 4 * y * 2)] = 0xff;
      
    }
  }
  
  e = cit_save_png (doubled,
                    FRAMEBUF_WIDTH * 2,
                    height,
                    png_output_filename,
                    CIT_PNG_SRC_TYPE_RGBA32,
                    1,
                    filesize_out,
                    FRAMEBUF_WIDTH * 4 * 2);

  return e;
  
}
